<?php


include 'db.php';
session_start();

if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
}
$CSRF = $_SESSION['csrf_token'];

// Helper: safe int percent clamp
function clamp_int($v, $min = 0, $max = 100) {
    $v = intval($v);
    if ($v < $min) return $min;
    if ($v > $max) return $max;
    return $v;
}

// CSV export (monthly cost report)
if (isset($_GET['export']) && $_GET['export'] === 'printers') {

    $month = $_GET['month'] ?? null; // YYYY-MM

    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=printers_report.csv');
    echo "\xEF\xBB\xBF"; // BOM for Excel

    $out = fopen('php://output', 'w');
    fputcsv($out, [
            'Printer ID',
            'Model',
            'Type',
            'Black (%)',
            'Drum (%)',
            'Developer (%)',
            'Status',
            'Next Maintenance',
            'Month',
            'Total Maintenance Cost'
    ]);

    // Base query
    $sql = "
        SELECT 
            p.id,
            p.model,
            p.printer_type,
            p.toner_black,
            p.drum_life,
            p.developer_life,
            p.status,
            p.next_maintenance_due,
            DATE_FORMAT(m.date_reported, '%Y-%m') AS month,
            COALESCE(SUM(m.cost), 0) AS total_cost
        FROM printers p
        LEFT JOIN printer_maintenance m 
            ON p.id = m.printer_id
    ";

    if ($month) {
        $month = $conn->real_escape_string($month);
        $sql .= " WHERE DATE_FORMAT(m.date_reported, '%Y-%m') = '$month'";
    }

    $sql .= "
        GROUP BY p.id, month
        ORDER BY month DESC, p.id ASC
    ";

    $res = $conn->query($sql);

    if (!$res) {
        die('Query failed: ' . $conn->error);
    }

    while ($r = $res->fetch_assoc()) {
        fputcsv($out, [
                $r['id'],
                $r['model'],
                $r['printer_type'],
                $r['toner_black'],
                $r['drum_life'],
                $r['developer_life'],
                $r['status'],
                $r['next_maintenance_due'],
                $r['month'] ?? $month ?? 'N/A',
                number_format($r['total_cost'], 2)
        ]);
    }

    fclose($out);
    exit();
}


// Initialize message
$successMsg = "";

// POST handlers
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Basic CSRF check (skip for CSV download)
    if (!isset($_POST['csrf']) || $_POST['csrf'] !== $CSRF) {
        http_response_code(403);
        exit('Invalid CSRF token');
    }

    // Printer form
    if (isset($_POST['action_printer'])) {
        $action = $_POST['action_printer'];
        $id = isset($_POST['printer_id']) ? intval($_POST['printer_id']) : null;
        $model = trim($_POST['model'] ?? '');
        $location = trim($_POST['location'] ?? '');
        $printer_type = ($_POST['printer_type'] ?? 'B/W') === 'Color' ? 'Color' : 'B/W';
        // If B/W, we'll store CMY as NULL
        $toner_cyan = ($printer_type === 'Color') ? clamp_int($_POST['toner_cyan'] ?? 0) : null;
        $toner_magenta = ($printer_type === 'Color') ? clamp_int($_POST['toner_magenta'] ?? 0) : null;
        $toner_yellow = ($printer_type === 'Color') ? clamp_int($_POST['toner_yellow'] ?? 0) : null;
        $toner_black = clamp_int($_POST['toner_black'] ?? 0);
        $drum = clamp_int($_POST['drum_life'] ?? 0);
        $developer = clamp_int($_POST['developer_life'] ?? 0);
        $status = in_array($_POST['status'] ?? '', ['Active','Maintenance','Inactive']) ? $_POST['status'] : 'Active';
        $last = $_POST['last_maintenance_date'] ?: null;
        $next = $_POST['next_maintenance_due'] ?: null;

        if ($action === 'add') {
            $sql = "INSERT INTO printers (model, location, printer_type, toner_cyan, toner_magenta, toner_yellow, toner_black, drum_life, developer_life, status, last_maintenance_date, next_maintenance_due)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param(
                    'sssiiiiissss',
                    $model,
                    $location,
                    $printer_type,
                    $toner_cyan,
                    $toner_magenta,
                    $toner_yellow,
                    $toner_black,
                    $drum,
                    $developer,
                    $status,
                    $last,
                    $next
            );

            $stmt->close();

            $sqlParts = [
                    'model' => $conn->real_escape_string($model),
                    'location' => $conn->real_escape_string($location),
                    'printer_type' => $conn->real_escape_string($printer_type),
                    'toner_cyan' => is_null($toner_cyan) ? "NULL" : intval($toner_cyan),
                    'toner_magenta' => is_null($toner_magenta) ? "NULL" : intval($toner_magenta),
                    'toner_yellow' => is_null($toner_yellow) ? "NULL" : intval($toner_yellow),
                    'toner_black' => intval($toner_black),
                    'drum' => intval($drum),
                    'developer' => intval($developer),
                    'status' => $conn->real_escape_string($status),
                    'last' => $last ? "'" . $conn->real_escape_string($last) . "'" : "NULL",
                    'next' => $next ? "'" . $conn->real_escape_string($next) . "'" : "NULL"
            ];
            $insertSql = "INSERT INTO printers (model, location, printer_type, toner_cyan, toner_magenta, toner_yellow, toner_black, drum_life, developer_life, status, last_maintenance_date, next_maintenance_due)
                VALUES ('{$sqlParts['model']}', '{$sqlParts['location']}', '{$sqlParts['printer_type']}', {$sqlParts['toner_cyan']}, {$sqlParts['toner_magenta']}, {$sqlParts['toner_yellow']}, {$sqlParts['toner_black']}, {$sqlParts['drum']}, {$sqlParts['developer']}, '{$sqlParts['status']}', {$sqlParts['last']}, {$sqlParts['next']})";
            $conn->query($insertSql);
            $_SESSION['successMsg'] = "Printer added successfully.";
        } else {
            // Edit
            $id_int = intval($id);
            $parts = [
                    "model='{$conn->real_escape_string($model)}'",
                    "location='{$conn->real_escape_string($location)}'",
                    "printer_type='{$conn->real_escape_string($printer_type)}'",
                    "toner_cyan=" . (is_null($toner_cyan) ? "NULL" : intval($toner_cyan)),
                    "toner_magenta=" . (is_null($toner_magenta) ? "NULL" : intval($toner_magenta)),
                    "toner_yellow=" . (is_null($toner_yellow) ? "NULL" : intval($toner_yellow)),
                    "toner_black=" . intval($toner_black),
                    "drum_life=" . intval($drum),
                    "developer_life=" . intval($developer),
                    "status='{$conn->real_escape_string($status)}'",
                    "last_maintenance_date=" . ($last ? "'" . $conn->real_escape_string($last) . "'" : "NULL"),
                    "next_maintenance_due=" . ($next ? "'" . $conn->real_escape_string($next) . "'" : "NULL")
            ];
            $sql = "UPDATE printers SET " . implode(', ', $parts) . " WHERE id = " . $id_int;
            $conn->query($sql);
            $_SESSION['successMsg'] = "Printer updated successfully.";
        }

        header("Location: " . $_SERVER['PHP_SELF']);
        exit();
    }

    // Maintenance form
    if (isset($_POST['action_maintenance'])) {
        $action = $_POST['action_maintenance'];
        $id = isset($_POST['maintenance_id']) ? intval($_POST['maintenance_id']) : null;
        $printer_id = intval($_POST['printer_id'] ?? 0);
        $issue = $conn->real_escape_string(trim($_POST['issue'] ?? ''));
        $description = $conn->real_escape_string(trim($_POST['description'] ?? ''));
        $technician = $conn->real_escape_string(trim($_POST['technician'] ?? ''));
        $status = in_array($_POST['status'] ?? '', ['Pending','Completed']) ? $_POST['status'] : 'Pending';
        $cost = floatval($_POST['cost'] ?? 0);
        $date_reported = $_POST['date_reported'] ?: null;
        $date_fixed = $_POST['date_fixed'] ?: null;

        if ($action === 'add') {
            $sql = "INSERT INTO printer_maintenance (printer_id, issue, description, technician, status, cost, date_reported, date_fixed)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param('issssdds', $printer_id, $issue, $description, $technician, $status, $cost, $date_reported, $date_fixed);
            $stmt->execute();
            $stmt->close();
            $_SESSION['successMsg'] = "Maintenance record added successfully.";
        } else {
            $id_int = intval($id);
            $sql = "UPDATE printer_maintenance SET printer_id=?, issue=?, description=?, technician=?, status=?, cost=?, date_reported=?, date_fixed=? WHERE id=?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param('issssdssi', $printer_id, $issue, $description, $technician, $status, $cost, $date_reported, $date_fixed, $id_int);
            $stmt->execute();
            $stmt->close();
            $_SESSION['successMsg'] = "Maintenance record updated successfully.";
        }

        header("Location: " . $_SERVER['PHP_SELF']);
        exit();
    }

    // Delete handlers via POST (we'll accept deletes through GET too but recommend POST)
    if (isset($_POST['delete_printer_id'])) {
        $delId = intval($_POST['delete_printer_id']);
        $stmt = $conn->prepare("DELETE FROM printers WHERE id = ?");
        $stmt->bind_param('i', $delId);
        $stmt->execute();
        $stmt->close();
        $_SESSION['successMsg'] = "Printer deleted successfully.";
        header("Location: " . $_SERVER['PHP_SELF']);
        exit();
    }

    if (isset($_POST['delete_maintenance_id'])) {
        $delId = intval($_POST['delete_maintenance_id']);
        $stmt = $conn->prepare("DELETE FROM printer_maintenance WHERE id = ?");
        $stmt->bind_param('i', $delId);
        $stmt->execute();
        $stmt->close();
        $_SESSION['successMsg'] = "Maintenance record deleted successfully.";
        header("Location: " . $_SERVER['PHP_SELF']);
        exit();
    }
}

// Support legacy GET deletes (kept but recommend POST)
if (isset($_GET['delete_printer'])) {
    $delId = intval($_GET['delete_printer']);
    $conn->query("DELETE FROM printers WHERE id = $delId");
    $_SESSION['successMsg'] = "Printer deleted successfully.";
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}
if (isset($_GET['delete_maintenance'])) {
    $delId = intval($_GET['delete_maintenance']);
    $conn->query("DELETE FROM printer_maintenance WHERE id = $delId");
    $_SESSION['successMsg'] = "Maintenance record deleted successfully.";
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}

// Fetch data for display
$printersRes = $conn->query("SELECT * FROM printers ORDER BY id DESC");
$printers = [];
while ($r = $printersRes->fetch_assoc()) $printers[] = $r;

$maintenanceRes = $conn->query("SELECT m.*, p.model as printer_model, p.id as printer_id FROM printer_maintenance m JOIN printers p ON m.printer_id=p.id ORDER BY m.id DESC");
$maintenanceLogs = [];
while ($m = $maintenanceRes->fetch_assoc()) $maintenanceLogs[] = $m;

$totalPrinters = count($printers);
$maintenanceDueRes = $conn->query("SELECT COUNT(*) as cnt FROM printers WHERE next_maintenance_due <= CURDATE()");
$maintenanceDue = ($maintenanceDueRes && $row = $maintenanceDueRes->fetch_assoc()) ? intval($row['cnt']) : 0;

// Data arrays for charts
$printerNames = [];
$tonerCyan = [];
$tonerMagenta = [];
$tonerYellow = [];
$tonerBlack = [];
$drumLife = [];
$developerLife = [];

foreach ($printers as $p) {
    $printerNames[] = $p['model'] ?: ("Printer " . $p['id']);
    $tonerCyan[] = is_null($p['toner_cyan']) ? null : intval($p['toner_cyan']);
    $tonerMagenta[] = is_null($p['toner_magenta']) ? null : intval($p['toner_magenta']);
    $tonerYellow[] = is_null($p['toner_yellow']) ? null : intval($p['toner_yellow']);
    $tonerBlack[] = intval($p['toner_black']);
    $drumLife[] = intval($p['drum_life']);
    $developerLife[] = intval($p['developer_life']);
}

// Maintenance costs per printer
$costsRes = $conn->query("SELECT p.model, COALESCE(SUM(m.cost), 0) as total_cost FROM printers p LEFT JOIN printer_maintenance m ON p.id=m.printer_id GROUP BY p.id ORDER BY p.id DESC");
$printerCostNames = [];
$printerCosts = [];
while ($c = $costsRes->fetch_assoc()) {
    $printerCostNames[] = $c['model'] ?: 'Unnamed';
    $printerCosts[] = floatval($c['total_cost']);
}

// Get success message (flash)
if (isset($_SESSION['successMsg'])) {
    $successMsg = $_SESSION['successMsg'];
    unset($_SESSION['successMsg']);
}

// Helper to JSON encode safely for data attributes
function safe_json_attr($arr) {
    return htmlspecialchars(json_encode($arr, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES), ENT_QUOTES, 'UTF-8');
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Printers Dashboard</title>
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .toner-bar { height: 22px; border-radius: 10px; overflow: hidden; background: #e9ecef; box-shadow: inset 0 -1px 0 rgba(0,0,0,.08); }
        .toner-fill { height: 100%; display:flex; align-items:center; padding-left:8px; font-weight:600; color:white; border-radius:10px; font-size:.85rem; white-space:nowrap; }
        .toner-fill.light-text-black { color: black; }
        .developer-fill { background: #343a40; }
    </style>
</head>
<body class="d-flex flex-column min-vh-100">
<nav class="navbar navbar-expand-lg navbar-light bg-white shadow-sm mb-4 border rounded">
    <div class="container-fluid">
        <a class="navbar-brand fw-bold text-primary" href="#">Printer Dashboard</a>
        <div class="d-flex ms-auto align-items-center">
            <span class="navbar-text me-3">Welcome, <strong><?= htmlspecialchars($_SESSION['user_name'] ?? 'User') ?></strong></span>
            <a href="logout.php" class="btn btn-outline-primary btn-sm">Logout</a>
        </div>
    </div>
</nav>

<div class="container-fluid">
    <?php if ($successMsg): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?= htmlspecialchars($successMsg) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <div class="row g-3 mb-3">
        <div class="col-md-4"><div class="card text-center p-2"><h6 class="mb-0">Total Printers</h6><p class="fs-3 mb-0"><?= $totalPrinters ?></p></div></div>
        <div class="col-md-4"><div class="card text-center p-2"><h6 class="mb-0">Maintenance Due</h6><p class="fs-3 mb-0"><?= $maintenanceDue ?></p></div></div>
        <div class="col-md-4 d-flex align-items-center justify-content-end">
            <form method="GET" class="d-flex align-items-center" style="gap:8px;">
                <input type="hidden" name="export" value="printers">
                <input type="month" name="month" class="form-control form-control-sm" value="<?= date('Y-m') ?>">
                <button type="submit" class="btn btn-outline-secondary btn-sm">Export CSV</button>
            </form>
        </div>
    </div>

    <ul class="nav nav-tabs mb-3">
        <li class="nav-item"><button class="nav-link active" data-bs-toggle="tab" data-bs-target="#printers_tab">Printers</button></li>
        <li class="nav-item"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#maintenance_tab">Maintenance</button></li>
        <li class="nav-item"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#reports_tab">Reports</button></li>
    </ul>

    <div class="tab-content">
        <!-- Printers -->
        <div class="tab-pane fade show active" id="printers_tab">
            <div class="d-flex justify-content-between align-items-center mb-2">
                <h5 class="mb-0">Printers</h5>
                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#printerModal" id="addPrinterBtn">Add Printer</button>
            </div>

            <div class="table-responsive">
                <table class="table table-striped align-middle">
                    <thead>
                    <tr>
                        <th>ID</th><th>Model</th><th>Location</th><th>Type</th><th>C (%)</th><th>M (%)</th><th>Y (%)</th><th>K (%)</th><th>Drum (%)</th><th>Developer (%)</th><th>Status</th><th>Next Maintenance</th><th>Actions</th>
                    </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($printers as $p):
                        $data = [
                                'id' => (int)$p['id'],
                                'model' => $p['model'],
                                'location' => $p['location'],
                                'printer_type' => $p['printer_type'],
                                'toner_cyan' => is_null($p['toner_cyan']) ? 0 : intval($p['toner_cyan']),
                                'toner_magenta' => is_null($p['toner_magenta']) ? 0 : intval($p['toner_magenta']),
                                'toner_yellow' => is_null($p['toner_yellow']) ? 0 : intval($p['toner_yellow']),
                                'toner_black' => intval($p['toner_black']),
                                'drum_life' => intval($p['drum_life']),
                                'developer_life' => intval($p['developer_life']),
                                'status' => $p['status'],
                                'last_maintenance_date' => $p['last_maintenance_date'],
                                'next_maintenance_due' => $p['next_maintenance_due']
                        ];
                        $dataAttr = safe_json_attr($data);
                        ?>
                        <tr>
                            <td><?= $p['id'] ?></td>
                            <td><?= htmlspecialchars($p['model']) ?></td>
                            <td><?= htmlspecialchars($p['location']) ?></td>
                            <td><?= htmlspecialchars($p['printer_type']) ?></td>

                            <?php if ($p['printer_type'] === 'Color'): ?>
                                <td style="min-width:160px;">
                                    <div class="toner-bar">
                                        <div class="toner-fill light-text-black" style="width: <?= intval($p['toner_cyan']) ?>%; background: #00FFFF;"><?= intval($p['toner_cyan']) ?>%</div>
                                    </div>
                                </td>
                                <td style="min-width:160px;">
                                    <div class="toner-bar">
                                        <div class="toner-fill" style="width: <?= intval($p['toner_magenta']) ?>%; background: #FF00FF;"><?= intval($p['toner_magenta']) ?>%</div>
                                    </div>
                                </td>
                                <td style="min-width:160px;">
                                    <div class="toner-bar">
                                        <div class="toner-fill light-text-black" style="width: <?= intval($p['toner_yellow']) ?>%; background: #FFFF00;"><?= intval($p['toner_yellow']) ?>%</div>
                                    </div>
                                </td>
                            <?php else: ?>
                                <td>-</td><td>-</td><td>-</td>
                            <?php endif; ?>

                            <td style="min-width:160px;">
                                <div class="toner-bar">
                                    <div class="toner-fill" style="width: <?= intval($p['toner_black']) ?>%; background: #000000;"><?= intval($p['toner_black']) ?>%</div>
                                </div>
                            </td>

                            <td style="min-width:160px;">
                                <div class="toner-bar">
                                    <div class="toner-fill" style="width: <?= intval($p['drum_life']) ?>%; background: #6c757d;"><?= intval($p['drum_life']) ?>%</div>
                                </div>
                            </td>

                            <td style="min-width:160px;">
                                <div class="toner-bar">
                                    <div class="toner-fill developer-fill" style="width: <?= intval($p['developer_life']) ?>%;"><?= intval($p['developer_life']) ?>%</div>
                                </div>
                            </td>

                            <td><?= htmlspecialchars($p['status']) ?></td>
                            <td><?= htmlspecialchars($p['next_maintenance_due']) ?></td>
                            <td>
                                <button class="btn btn-sm btn-warning me-1 edit-printer-btn" data-printer='<?= $dataAttr ?>' data-bs-toggle="modal" data-bs-target="#printerModal">Edit</button>

                                <!-- Delete via small form (POST) to be safer -->
                                <form method="POST" style="display:inline;" onsubmit="return confirm('Delete printer?');">
                                    <input type="hidden" name="csrf" value="<?= $CSRF ?>">
                                    <input type="hidden" name="delete_printer_id" value="<?= intval($p['id']) ?>">
                                    <button class="btn btn-sm btn-danger">Delete</button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Maintenance -->
        <div class="tab-pane fade" id="maintenance_tab">
            <div class="d-flex justify-content-between align-items-center mb-2">
                <h5 class="mb-0">Maintenance Logs</h5>
                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#maintenanceModal" id="addMaintenanceBtn">Add Maintenance</button>
            </div>

            <div class="table-responsive">
                <table class="table table-striped align-middle">
                    <thead>
                    <tr><th>Printer</th><th>Issue</th><th>Description</th><th>Technician</th><th>Status</th><th>Cost</th><th>Reported</th><th>Fixed</th><th>Actions</th></tr>
                    </thead>
                    <tbody>
                    <?php foreach ($maintenanceLogs as $m): ?>
                        <tr>
                            <td><?= htmlspecialchars($m['printer_model']) ?></td>
                            <td><?= htmlspecialchars($m['issue']) ?></td>
                            <td><?= htmlspecialchars($m['description']) ?></td>
                            <td><?= htmlspecialchars($m['technician']) ?></td>
                            <td><?= htmlspecialchars($m['status']) ?></td>
                            <td><?= number_format($m['cost'],2) ?></td>
                            <td><?= htmlspecialchars($m['date_reported']) ?></td>
                            <td><?= htmlspecialchars($m['date_fixed']) ?></td>
                            <td>
                                <button class="btn btn-sm btn-warning edit-maint-btn" data-maint='<?= safe_json_attr($m) ?>' data-bs-toggle="modal" data-bs-target="#maintenanceModal">Edit</button>

                                <form method="POST" style="display:inline;" onsubmit="return confirm('Delete maintenance record?');">
                                    <input type="hidden" name="csrf" value="<?= $CSRF ?>">
                                    <input type="hidden" name="delete_maintenance_id" value="<?= intval($m['id']) ?>">
                                    <button class="btn btn-sm btn-danger">Delete</button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Reports -->
        <div class="tab-pane fade" id="reports_tab">


            <div class="row mb-4">
                <div class="col-md-6"><canvas id="tonerChart"></canvas></div>
                <div class="col-md-6"><canvas id="maintenanceCostChart"></canvas></div>
            </div>


        </div>
    </div>
</div>

<!-- Printer Modal -->
<div class="modal fade" id="printerModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <form method="POST" class="modal-content" id="printerForm">
            <div class="modal-header">
                <h5 class="modal-title" id="printerModalTitle">Printer</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <input type="hidden" name="csrf" value="<?= $CSRF ?>">
                <input type="hidden" name="printer_id" id="printer_id" value="">
                <input type="hidden" name="action_printer" id="action_printer" value="add">

                <div class="mb-2">
                    <label class="form-label">Model</label>
                    <input type="text" name="model" id="model" class="form-control" required>
                </div>

                <div class="mb-2">
                    <label class="form-label">Location</label>
                    <input type="text" name="location" id="location" class="form-control">
                </div>

                <div class="mb-2">
                    <label class="form-label">Type</label>
                    <select name="printer_type" id="printer_type" class="form-select" required>
                        <option value="B/W">B/W</option>
                        <option value="Color">Color</option>
                    </select>
                </div>

                <div id="cmys_group">
                    <div class="mb-2">
                        <label class="form-label">Toner Cyan (%)</label>
                        <input type="number" name="toner_cyan" id="toner_cyan" class="form-control" min="0" max="100" value="0">
                    </div>
                    <div class="mb-2">
                        <label class="form-label">Toner Magenta (%)</label>
                        <input type="number" name="toner_magenta" id="toner_magenta" class="form-control" min="0" max="100" value="0">
                    </div>
                    <div class="mb-2">
                        <label class="form-label">Toner Yellow (%)</label>
                        <input type="number" name="toner_yellow" id="toner_yellow" class="form-control" min="0" max="100" value="0">
                    </div>
                </div>

                <div class="mb-2">
                    <label class="form-label">Toner Black (%)</label>
                    <input type="number" name="toner_black" id="toner_black" class="form-control" min="0" max="100" value="0">
                </div>

                <div class="mb-2">
                    <label class="form-label">Drum Life (%)</label>
                    <input type="number" name="drum_life" id="drum_life" class="form-control" min="0" max="100" value="0">
                </div>

                <div class="mb-2">
                    <label class="form-label">Developer Life (%)</label>
                    <input type="number" name="developer_life" id="developer_life" class="form-control" min="0" max="100" value="0">
                </div>

                <div class="mb-2">
                    <label class="form-label">Status</label>
                    <select name="status" id="status" class="form-select">
                        <option>Active</option>
                        <option>Maintenance</option>
                        <option>Inactive</option>
                    </select>
                </div>

                <div class="mb-2 row">
                    <div class="col">
                        <label class="form-label">Last Maintenance</label>
                        <input type="date" name="last_maintenance_date" id="last_maintenance_date" class="form-control">
                    </div>
                    <div class="col">
                        <label class="form-label">Next Maintenance Due</label>
                        <input type="date" name="next_maintenance_due" id="next_maintenance_due" class="form-control">
                    </div>
                </div>
            </div>

            <div class="modal-footer">
                <button type="submit" class="btn btn-success">Save</button>
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            </div>
        </form>
    </div>
</div>

<!-- Maintenance Modal -->
<div class="modal fade" id="maintenanceModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <form method="POST" class="modal-content" id="maintenanceForm">
            <div class="modal-header">
                <h5 class="modal-title">Maintenance</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <input type="hidden" name="csrf" value="<?= $CSRF ?>">
                <input type="hidden" name="maintenance_id" id="maintenance_id" value="">
                <input type="hidden" name="action_maintenance" id="action_maintenance" value="add">

                <div class="mb-2">
                    <label class="form-label">Printer</label>
                    <select name="printer_id" id="maintenance_printer_id" class="form-select" required>
                        <option value="">Select Printer</option>
                        <?php foreach ($printers as $pp): ?>
                            <option value="<?= intval($pp['id']) ?>"><?= htmlspecialchars($pp['model']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="mb-2">
                    <label class="form-label">Issue</label>
                    <input type="text" name="issue" id="issue" class="form-control">
                </div>

                <div class="mb-2">
                    <label class="form-label">Description</label>
                    <textarea name="description" id="description" class="form-control" rows="3"></textarea>
                </div>

                <div class="mb-2">
                    <label class="form-label">Technician</label>
                    <input type="text" name="technician" id="technician" class="form-control">
                </div>

                <div class="mb-2">
                    <label class="form-label">Status</label>
                    <select name="status" id="maintenance_status" class="form-select">
                        <option>Pending</option>
                        <option>Completed</option>
                    </select>
                </div>

                <div class="mb-2">
                    <label class="form-label">Cost</label>
                    <input type="number" step="0.01" name="cost" id="cost" class="form-control">
                </div>

                <div class="mb-2 row">
                    <div class="col">
                        <label class="form-label">Date Reported</label>
                        <input type="date" name="date_reported" id="date_reported" class="form-control">
                    </div>
                    <div class="col">
                        <label class="form-label">Date Fixed</label>
                        <input type="date" name="date_fixed" id="date_fixed" class="form-control">
                    </div>
                </div>
            </div>

            <div class="modal-footer">
                <button type="submit" class="btn btn-success">Save</button>
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            </div>
        </form>
    </div>
</div>

<footer class="bg-light text-center text-muted py-3 mt-auto border-top">
    &copy; <?= date('Y') ?> BroadHorizonGroup
</footer>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script src="https://cdn.jsdelivr.net/npm/chartjs-plugin-annotation@2.2.1/dist/chartjs-plugin-annotation.min.js"></script>
<script>
    // Page data
    const threshold = 20;
    const printerNames = <?= json_encode($printerNames) ?>;
    const tonerCyan = <?= json_encode($tonerCyan) ?>;
    const tonerMagenta = <?= json_encode($tonerMagenta) ?>;
    const tonerYellow = <?= json_encode($tonerYellow) ?>;
    const tonerBlack = <?= json_encode($tonerBlack) ?>;
    const drumLife = <?= json_encode($drumLife) ?>;
    const developerLife = <?= json_encode($developerLife) ?>;
    const printerCostNames = <?= json_encode($printerCostNames) ?>;
    const printerCosts = <?= json_encode($printerCosts) ?>;

    // Color helper
    function colorFor(value, defaultColor) {
        if (value === null || value === undefined) return '#cccccc';
        return (value < threshold) ? 'rgba(220,53,69,0.9)' : defaultColor;
    }

    // Toner chart with threshold line
    new Chart(document.getElementById('tonerChart'), {
        type: 'bar',
        data: {
            labels: printerNames,
            datasets: [
                { label: 'Cyan', data: tonerCyan.map(v => v === null ? 0 : v), backgroundColor: tonerCyan.map(v => colorFor(v, '#00FFFF')) },
                { label: 'Magenta', data: tonerMagenta.map(v => v === null ? 0 : v), backgroundColor: tonerMagenta.map(v => colorFor(v, '#FF00FF')) },
                { label: 'Yellow', data: tonerYellow.map(v => v === null ? 0 : v), backgroundColor: tonerYellow.map(v => colorFor(v, '#FFFF00')) },
                { label: 'Black', data: tonerBlack, backgroundColor: tonerBlack.map(v => colorFor(v, '#000000')) },
                { label: 'Drum', data: drumLife, backgroundColor: drumLife.map(v => colorFor(v, '#6c757d')) },
                { label: 'Developer', data: developerLife, backgroundColor: developerLife.map(v => colorFor(v, '#343a40')) }
            ]
        },
        options: {
            responsive: true,
            plugins: {
                title: { display: true, text: 'Toner / Drum / Developer Levels' },
                annotation: {
                    annotations: {
                        thresholdLine: {
                            type: 'line',
                            yMin: threshold,
                            yMax: threshold,
                            borderColor: 'red',
                            borderWidth: 2,
                            borderDash: [6, 4],
                            label: {
                                content: `Threshold (${threshold}%)`,
                                enabled: true,
                                position: 'end',
                                backgroundColor: 'rgba(220,53,69,0.7)',
                                color: 'white',
                                font: { weight: 'bold' }
                            }
                        }
                    }
                }
            },
            scales: { y: { min: 0, max: 100 } }
        },
        plugins: [Chart.registry.getPlugin('annotation')]
    });

    // Maintenance cost chart
    new Chart(document.getElementById('maintenanceCostChart'), {
        type: 'bar',
        data: {
            labels: printerCostNames,
            datasets: [{ label: 'Maintenance Cost', data: printerCosts, backgroundColor: printerCosts.map(()=> '#0d6efd') }]
        },
        options: { responsive: true, plugins: { title: { display: true, text: 'Maintenance Costs per Printer' } }, scales: { y: { beginAtZero: true } } }
    });

    // UI interactions
    function updateCMYVisibility() {
        const type = document.getElementById('printer_type').value;
        const group = document.getElementById('cmys_group');
        if (type === 'Color') {
            group.style.display = '';
            document.getElementById('toner_cyan').disabled = false;
            document.getElementById('toner_magenta').disabled = false;
            document.getElementById('toner_yellow').disabled = false;
        } else {
            group.style.display = 'none';
            document.getElementById('toner_cyan').disabled = true;
            document.getElementById('toner_magenta').disabled = true;
            document.getElementById('toner_yellow').disabled = true;
        }
    }

    document.getElementById('printer_type')?.addEventListener('change', updateCMYVisibility);

    document.getElementById('addPrinterBtn')?.addEventListener('click', function(){
        document.getElementById('printerForm').reset();
        document.getElementById('action_printer').value = 'add';
        document.getElementById('printerModalTitle').textContent = 'Add Printer';
        document.getElementById('printer_type').value = 'B/W';
        document.getElementById('drum_life').value = 0;
        document.getElementById('developer_life').value = 0;
        updateCMYVisibility();
    });

    function parseJSON(s){ try { return JSON.parse(s); } catch(e) { return null; } }

    document.querySelectorAll('.edit-printer-btn').forEach(btn => {
        btn.addEventListener('click', function(){
            const raw = this.getAttribute('data-printer');
            const data = parseJSON(raw);
            if(!data) return;
            document.getElementById('action_printer').value = 'edit';
            document.getElementById('printerModalTitle').textContent = 'Edit Printer';
            document.getElementById('printer_id').value = data.id || '';
            document.getElementById('model').value = data.model || '';
            document.getElementById('location').value = data.location || '';
            document.getElementById('printer_type').value = data.printer_type || 'B/W';
            document.getElementById('toner_cyan').value = data.toner_cyan ?? 0;
            document.getElementById('toner_magenta').value = data.toner_magenta ?? 0;
            document.getElementById('toner_yellow').value = data.toner_yellow ?? 0;
            document.getElementById('toner_black').value = data.toner_black ?? 0;
            document.getElementById('drum_life').value = data.drum_life ?? 0;
            document.getElementById('developer_life').value = data.developer_life ?? 0;
            document.getElementById('status').value = data.status || 'Active';
            document.getElementById('last_maintenance_date').value = data.last_maintenance_date || '';
            document.getElementById('next_maintenance_due').value = data.next_maintenance_due || '';
            updateCMYVisibility();
        });
    });

    document.getElementById('addMaintenanceBtn')?.addEventListener('click', function(){
        document.getElementById('maintenanceForm').reset();
        document.getElementById('action_maintenance').value = 'add';
        document.querySelector('#maintenanceModal .modal-title').textContent = 'Add Maintenance';
    });

    document.querySelectorAll('.edit-maint-btn').forEach(btn => {
        btn.addEventListener('click', function(){
            const raw = this.getAttribute('data-maint');
            const data = parseJSON(raw);
            if(!data) return;
            document.getElementById('action_maintenance').value = 'edit';
            document.getElementById('maintenance_id').value = data.id || '';
            document.getElementById('maintenance_printer_id').value = data.printer_id || '';
            document.getElementById('issue').value = data.issue || '';
            document.getElementById('description').value = data.description || '';
            document.getElementById('technician').value = data.technician || '';
            document.getElementById('maintenance_status').value = data.status || 'Pending';
            document.getElementById('cost').value = data.cost || '';
            document.getElementById('date_reported').value = data.date_reported || '';
            document.getElementById('date_fixed').value = data.date_fixed || '';
            document.querySelector('#maintenanceModal .modal-title').textContent = 'Edit Maintenance';
        });
    });

    // ensure CMY visibility initial state on page load
    document.addEventListener('DOMContentLoaded', function(){ updateCMYVisibility(); });
</script>
