<?php
session_start();
$mysqli = new mysqli("localhost", "root", "", "mas_dashboard");
if ($mysqli->connect_errno) die("DB Connection failed: " . $mysqli->connect_error);

if (!isset($_SESSION['staff_id'])) {
    header("Location: login.php");
    exit;
}

$staff_id = $_SESSION['staff_id'];
$branch = $_SESSION['branch'];

$job_card_id = 0;
if (isset($_GET['job_card_id']) && is_numeric($_GET['job_card_id'])) {
    $job_card_id = intval($_GET['job_card_id']);
} elseif (isset($_GET['id']) && is_numeric($_GET['id'])) {
    $job_card_id = intval($_GET['id']);
}
if ($job_card_id <= 0) die("Invalid or missing job card ID.");

// Handle stage actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $stage_id = intval($_POST['stage_id']);
    $action = $_POST['action'];
    $remarks = isset($_POST['remarks']) ? substr(trim($_POST['remarks']), 0, 255) : '';

    if ($action === 'start') {
        $stmt = $mysqli->prepare("UPDATE job_card_stages SET status='In Progress', started_at=NOW() WHERE id=? AND FIND_IN_SET(?, assigned_to)");
        $stmt->bind_param("ii", $stage_id, $staff_id);
        $stmt->execute();
        $stmt->close();
    } elseif ($action === 'done') {
        $stmt = $mysqli->prepare("UPDATE job_card_stages SET status='Completed', completed_at=NOW(), remarks=? WHERE id=? AND FIND_IN_SET(?, assigned_to)");
        $stmt->bind_param("sii", $remarks, $stage_id, $staff_id);
        $stmt->execute();
        $stmt->close();
    }
    header("Location: view_job_card.php?job_card_id=" . $job_card_id);
    exit;
}

// Fetch job card
$stmt = $mysqli->prepare("SELECT job_card_number, customer_name, order_description, branch, status FROM job_cards WHERE id=? AND branch=?");
$stmt->bind_param("is", $job_card_id, $branch);
$stmt->execute();
$stmt->bind_result($job_card_number, $customer_name, $order_description, $jc_branch, $status);
if (!$stmt->fetch()) die("Job card not found or access denied.");
$stmt->close();

// Fetch stages
$stage_stmt = $mysqli->prepare("
    SELECT id, stage_name, status, assigned_to, started_at, completed_at, remarks, TIMESTAMPDIFF(HOUR, started_at, NOW()) as hours_since_start
    FROM job_card_stages
    WHERE job_card_id=?
    ORDER BY id ASC
");
$stage_stmt->bind_param("i", $job_card_id);
$stage_stmt->execute();
$stages_result = $stage_stmt->get_result();
$stages = $stages_result->fetch_all(MYSQLI_ASSOC);
$stage_stmt->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Job Card Pipeline Responsive</title>
    <style>
        body { font-family: Arial, sans-serif; background:#f0f4f8; padding:20px; }
        h2 { color:#30307d; margin-bottom:20px; }

        /* Pipeline container */
        .pipeline-wrapper {
            position: relative;
            margin-top:50px;
            overflow-x:auto;
            white-space: nowrap;
            padding-bottom:50px;
            display:flex;
            align-items:center;
        }

        /* Stage card */
        .stage-card {
            display:inline-block;
            width:150px;
            min-width:150px;
            padding:15px;
            border-radius:12px;
            background:#fff;
            box-shadow:0 3px 6px rgba(0,0,0,0.2);
            text-align:center;
            position:relative;
            margin-right:40px;
            transition: transform 0.3s, background 0.3s;
        }
        .stage-card:hover { transform: scale(1.08); }

        /* Stage states */
        .assigned { border:3px solid #27ae60; }
        .completed { background:#d1f0d1; }
        .locked { background:#f9d6d5; color:#a94442; opacity:0.7; }
        .overdue { border:3px solid red; font-weight:bold; }

        /* Tooltip */
        .stage-card .tooltip {
            visibility:hidden;
            width:180px;
            background:#333;
            color:#fff;
            text-align:left;
            padding:8px;
            border-radius:6px;
            position:absolute;
            bottom:110%;
            left:50%;
            transform: translateX(-50%);
            z-index:10;
            font-size:12px;
            opacity:0;
            transition: opacity 0.3s;
        }
        .stage-card:hover .tooltip { visibility:visible; opacity:1; }

        /* Arrow */
        .arrow {
            display:inline-block;
            font-size:24px;
            color:#30307d;
            vertical-align:middle;
            margin-right:40px;
        }

        /* Progress bar */
        .progress-bar-container {
            position:absolute;
            height:8px;
            background:#ddd;
            top:50px;
            left:0;
            right:0;
            z-index:0;
            border-radius:4px;
        }
        .progress-fill {
            height:8px;
            background:#27ae60;
            border-radius:4px;
            width:0;
            transition: width 1s ease-in-out;
            position:absolute;
            top:0;
            left:0;
        }

        /* Buttons */
        .action-btn { padding:5px 8px; border:none; border-radius:5px; cursor:pointer; font-size:12px; margin-top:5px; }
        .start-btn { background:#f39c12; color:white; }
        .done-btn { background:#27ae60; color:white; }

        /* Responsive vertical layout */
        @media(max-width:700px) {
            .pipeline-wrapper {
                flex-direction:column;
                white-space:normal;
                align-items:flex-start;
            }
            .stage-card { margin-right:0; margin-bottom:40px; }
            .arrow { transform: rotate(90deg); display:block; margin:10px 0; }
            .progress-bar-container { top:auto; left:50%; width:8px; height: calc(100% - 50px); transform: translateX(-50%); }
            .progress-fill { width:100%; height:0; top:0; left:0; }
        }
    </style>
</head>
<body>

<h2>Job Card: <?= htmlspecialchars($job_card_number) ?></h2>
<p><strong>Customer:</strong> <?= htmlspecialchars($customer_name) ?></p>
<p><strong>Order Description:</strong> <?= htmlspecialchars($order_description) ?></p>
<p><strong>Status:</strong> <?= htmlspecialchars($status) ?></p>
<p><strong>Branch:</strong> <?= htmlspecialchars($jc_branch) ?></p>

<div class="pipeline-wrapper">
    <div class="progress-bar-container">
        <div class="progress-fill" id="progress-fill"></div>
    </div>

    <?php
    $previous_done = true;
    foreach($stages as $index => $stage):
        $assigned = array_map('intval', explode(",",$stage['assigned_to']));
        $is_assigned = in_array($staff_id,$assigned);
        $classes=[];
        if($stage['status']==='Completed') $classes[]='completed';
        elseif(!$previous_done) $classes[]='locked';
        elseif($stage['hours_since_start']>=24) $classes[]='overdue';
        if($is_assigned) $classes[]='assigned';
        ?>
        <div class="stage-card <?=implode(' ',$classes)?>">
            <div class="stage-name"><?= htmlspecialchars($stage['stage_name']) ?></div>
            <div class="stage-status"><?= htmlspecialchars($stage['status']) ?></div>

            <?php if($is_assigned && $stage['status']==='Pending' && $previous_done): ?>
                <form method="post">
                    <input type="hidden" name="stage_id" value="<?= $stage['id'] ?>">
                    <input type="hidden" name="action" value="start">
                    <button class="action-btn start-btn">Start</button>
                </form>
            <?php elseif($is_assigned && $stage['status']==='In Progress'): ?>
                <form method="post">
                    <input type="hidden" name="stage_id" value="<?= $stage['id'] ?>">
                    <input type="hidden" name="action" value="done">
                    <input type="text" name="remarks" placeholder="Remarks" required><br>
                    <button class="action-btn done-btn">Done</button>
                </form>
            <?php elseif(!$is_assigned || !$previous_done): ?>
                <div style="opacity:0.6;">Locked</div>
            <?php endif; ?>

            <div class="tooltip">
                Assigned To: <?= htmlspecialchars($stage['assigned_to']) ?><br>
                Started At: <?= htmlspecialchars($stage['started_at']) ?><br>
                Completed At: <?= htmlspecialchars($stage['completed_at']) ?><br>
                Remarks: <?= htmlspecialchars($stage['remarks']) ?>
            </div>
        </div>

        <?php if($index<count($stages)-1) echo '<span class="arrow">&#8594;</span>'; ?>
        <?php $previous_done = ($stage['status']==='Completed'); endforeach; ?>
</div>

<script>
    function updateProgressBar() {
        let completed = <?= count(array_filter($stages, fn($s)=>$s['status']==='Completed')) ?>;
        let total = <?= count($stages) ?>;
        let percent = (completed / total) * 100;

        let progressFill = document.getElementById('progress-fill');
        let pipeline = document.querySelector('.pipeline-wrapper');

        // Get scrollable width
        let scrollWidth = pipeline.scrollWidth - pipeline.clientWidth;

        // Set progress width relative to scrollable width
        progressFill.style.width = `calc(${percent}% + ${scrollWidth * (percent/100)}px)`;
    }

    window.addEventListener('resize', updateProgressBar);
    window.addEventListener('load', updateProgressBar);
</script>

</body>
</html>
