<?php
session_start();
include 'db.php';

// Ensure user logged in with branch info
$user_branch = $_SESSION['branch'] ?? null;
if (!$user_branch) die("Branch information missing. Access denied.");

// Validate job card ID
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) die("Invalid job card ID.");
$job_card_id = (int)$_GET['id'];

// ---------------- DELIVERY NOTE CHECK ----------------
$noteStmt = $conn->prepare("
    SELECT dn.note_file 
    FROM delivery_notes dn 
    JOIN job_cards jc ON dn.job_card_id = jc.id 
    WHERE dn.job_card_id = ? AND jc.branch = ? LIMIT 1
");
$noteStmt->bind_param("is", $job_card_id, $user_branch);
$noteStmt->execute();
$noteResult = $noteStmt->get_result()->fetch_assoc();
$noteStmt->close();

$deliveryNoteExists = !empty($noteResult['note_file']);
$deliveryNotePath = $deliveryNoteExists ? $noteResult['note_file'] : '';

// ---------------- DELIVERY UPDATE ----------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delivery_update'])) {
    $deliver_now = (int)($_POST['deliver_now'] ?? 0);
    if ($deliver_now > 0) {
        $conn->begin_transaction();
        try {
            $stmt = $conn->prepare("SELECT delivered_quantity, total_order_quantity FROM job_cards WHERE id = ? AND branch = ? FOR UPDATE");
            $stmt->bind_param("is", $job_card_id, $user_branch);
            $stmt->execute();
            $result = $stmt->get_result()->fetch_assoc();
            $stmt->close();

            if ($result) {
                $new_qty = (int)$result['delivered_quantity'] + $deliver_now;
                if ($new_qty <= (int)$result['total_order_quantity']) {
                    $upd = $conn->prepare("UPDATE job_cards SET delivered_quantity = ? WHERE id = ? AND branch = ?");
                    $upd->bind_param("iis", $new_qty, $job_card_id, $user_branch);
                    $upd->execute();
                    $upd->close();
                    $_SESSION['message'] = "Delivery updated successfully.";
                } else {
                    $_SESSION['error'] = "Cannot deliver more than total ordered quantity.";
                }
            } else {
                $_SESSION['error'] = "Job card not found or access denied.";
            }
            $conn->commit();
        } catch (Exception $e) {
            $conn->rollback();
            $_SESSION['error'] = "Delivery update failed.";
        }
        header("Location: job_card.php?id=$job_card_id");
        exit;
    }
}

// ---------------- FETCH JOB CARD ----------------
$stmt = $conn->prepare("SELECT * FROM job_cards WHERE id = ? AND branch = ?");
$stmt->bind_param("is", $job_card_id, $user_branch);
$stmt->execute();
$job = $stmt->get_result()->fetch_assoc();
$stmt->close();
if (!$job) die("Job card not found or you do not have access to this branch.");

$department = strtolower(trim($job['department'] ?? ''));

// ---------------- STAGES ----------------
$commercialBrandingProcesses = [
    'Dye Sublimation' => ['Artwork','Block Resize','Pattern','Printing from Sublimation','Pressing','Sewing','Finishing & Packaging'],
    'Embroidery' => ['Logo Setup','Embroidery','Packaging & Finishing'],
    'Engraving' => ['Logo Setup','Engraving','Finishing & Packaging'],
];
$departmentStages = [
    'signage' => ['Design','Chromadek Cutting','Fabrication','Artwork Pasting','Installation'],
    'bulk printing' => ['Design','Plate Making','Printing','Perforating','Collating','Numbering','Cutting','Binding'],
];

$stages = ($department === 'commercial branding')
    ? ($commercialBrandingProcesses[$job['process_type'] ?? ''] ?? [])
    : ($departmentStages[$department] ?? []);

// ---------------- STAFF LIST ----------------
$deptLower = strtolower(trim($job['department'] ?? ''));
$staffStmt = $conn->prepare("SELECT id, employee_name FROM employees WHERE branch = ? AND LOWER(department) = ? AND is_supervisor = 0 ORDER BY employee_name ASC");
$staffStmt->bind_param("ss", $user_branch, $deptLower);
$staffStmt->execute();
$staffResult = $staffStmt->get_result();
$staffList = [];
while ($row = $staffResult->fetch_assoc()) {
    $staffList[$row['id']] = $row['employee_name'];
}
$staffStmt->close();

// ---------------- STAGE DATA ----------------
$stageDataMap = [];
if ($stages) {
    $placeholders = implode(',', array_fill(0, count($stages), '?'));
    $types = str_repeat('s', count($stages));
    $params = $stages;

    $query = "SELECT * FROM job_card_stages WHERE job_card_id = ? AND stage_name IN ($placeholders)";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("i$types", $job_card_id, ...$params);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $stageDataMap[$row['stage_name']] = $row;
    }
    $stmt->close();
}

// ---------------- CHECK STAGE COMPLETION ----------------
$allStagesCompleted = true;
foreach ($stages as $s) {
    if (!isset($stageDataMap[$s]) || strtolower($stageDataMap[$s]['status']) !== 'completed') {
        $allStagesCompleted = false;
        break;
    }
}

// ---------------- DELIVERY INFO ----------------
$totalOrderQty = (int)$job['total_order_quantity'];
$deliveredQty = (int)$job['delivered_quantity'];
$leftToDeliver = max(0, $totalOrderQty - $deliveredQty);
$fullyDelivered = ($deliveredQty >= $totalOrderQty);

// ---------------- DISPATCH ----------------
$stmt = $conn->prepare("SELECT dispatched, dispatched_at FROM job_cards WHERE id = ? AND branch = ?");
$stmt->bind_param("is", $job_card_id, $user_branch);
$stmt->execute();
$dispRes = $stmt->get_result()->fetch_assoc();
$stmt->close();

$job_dispatched = $dispRes['dispatched'] ?? 0;
$dispatched_at = $dispRes['dispatched_at'] ?? null;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['dispatch']) && $allStagesCompleted && !$job_dispatched && $fullyDelivered) {
    $stmt = $conn->prepare("UPDATE job_cards SET dispatched = 1, dispatched_at = NOW() WHERE id = ? AND branch = ?");
    $stmt->bind_param("is", $job_card_id, $user_branch);
    $stmt->execute();
    $stmt->close();
    $_SESSION['message'] = "Job dispatched successfully.";
    header("Location: view_job.php?id=$job_card_id");
    exit;
}


$message = $_SESSION['message'] ?? '';
$error = $_SESSION['error'] ?? '';
unset($_SESSION['message'], $_SESSION['error']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Job Card <?= htmlspecialchars($job['job_card_number']) ?></title>
    <link rel="stylesheet" href="styles.css">
    <style>
        body {font-family: Arial,sans-serif;margin:20px;}
        table {border-collapse: collapse;width:100%;margin-bottom:20px;font-size:13px;}
        th,td {border:1px solid #ccc;padding:6px 8px;vertical-align: middle;}
        th {background-color: #eee;}
        .btn {padding:6px 10px;font-size:12px;cursor:pointer;border-radius:4px;}
        .btn-primary {background:#007bff;color:white;border:none;}
        .btn-disabled {background:#ccc;color:#666;cursor:not-allowed;}
        .message {margin-bottom:20px;padding:10px;border-radius:4px;}
        .message.success {background:#d4edda;color:#155724;border:1px solid #c3e6cb;}
        .message.error {background:#f8d7da;color:#721c24;border:1px solid #f5c6cb;}
        details>summary {cursor:pointer;font-size:12px;font-weight:600;}
    </style>
</head>
<body>

<nav style="padding:10px 20px;color:white;background:#30307d;margin-bottom:20px;border-radius:6px;">
    <a href="Admins.php" style="color:white;text-decoration:none;margin-right:20px;font-weight:bold;">Dashboard</a>
    <a href="profile.php" style="color:white;text-decoration:none;margin-right:20px;font-weight:bold;">My Profile</a>
    <a href="logout.php" style="color:white;text-decoration:none;font-weight:bold;">Logout</a>
</nav>

<div style="display:flex;gap:20px;flex-wrap:wrap;margin-bottom:30px;">
    <div style="flex:1 1 300px;background:#fff;padding:20px;border-radius:10px;box-shadow:0 2px 10px rgba(0,0,0,0.1);">
        <h2 style="color:#30307d;margin-bottom:10px;">Job Card: <?= htmlspecialchars($job['job_card_number']) ?></h2>
        <p><strong>Branch:</strong> <?= htmlspecialchars($user_branch) ?></p>
        <p><strong>Order Number:</strong> <?= htmlspecialchars($job['job_order_id'] ?? 'N/A') ?></p>
        <p><strong>Customer:</strong> <?= htmlspecialchars($job['customer_name'] ?? 'N/A') ?></p>
        <p><strong>Description:</strong> <?= htmlspecialchars($job['order_description'] ?? 'N/A') ?></p>
        <p><strong>Status:</strong> <span style="color:<?= strtolower($job['status']??'')==='closed'?'green':'#e67e22' ?>; font-weight:bold"><?= htmlspecialchars($job['status']??'N/A') ?></span></p>
        <p><strong>Due Date:</strong> <?= htmlspecialchars($job['due_date']??'N/A') ?></p>
    </div>

    <div style="flex:1 1 300px;background:#fff;padding:20px;border-radius:10px;box-shadow:0 2px 10px rgba(0,0,0,0.1);">
        <h3 style="color:#2c3e50;">Delivery Details</h3>
        <p><strong>Ordered Quantity:</strong> <?= $totalOrderQty ?></p>
        <p><strong>Already Delivered:</strong> <?= $deliveredQty ?></p>
        <p><strong>Left to Deliver:</strong> <?= $leftToDeliver ?></p>
        <?php if($leftToDeliver===0):?><p style="color:green;font-weight:bold;">✔ All items delivered.</p><?php endif;?>
    </div>
</div>

<?php if ($message): ?><p class="message success"><?= htmlspecialchars($message) ?></p><?php endif;?>
<?php if ($error): ?><p class="message error"><?= htmlspecialchars($error) ?></p><?php endif;?>

<?php if (!$job_dispatched && !$fullyDelivered): ?>
    <form method="POST" style="margin-bottom:20px;">
        <label for="deliver_now">Deliver quantity now (max <?= $leftToDeliver ?>): </label>
        <input type="number" id="deliver_now" name="deliver_now" min="1" max="<?= $leftToDeliver ?>" required>
        <button type="submit" name="delivery_update" class="btn btn-primary">Update Delivery</button>
    </form>
<?php endif;?>

<h3>Stages</h3>
<table>
    <thead>
    <tr><th>Stage</th><th>Status</th><th>Assigned To</th><th>Start</th><th>End</th><th>Remarks</th><th>Action</th></tr>
    </thead>
    <tbody>
    <?php foreach($stages as $stage):
        $stageData = $stageDataMap[$stage] ?? [];
        $assigned = !empty($stageData['assigned_to']) ? explode(',', $stageData['assigned_to']) : [];
        $locked = !empty($assigned) || (isset($stageData['status']) && strtolower($stageData['status'])==='completed');
        ?>
        <tr>
            <form method="POST" action="assign_stage.php">
                <input type="hidden" name="job_card_id" value="<?= $job_card_id ?>">
                <input type="hidden" name="stage_name" value="<?= htmlspecialchars($stage) ?>">

                <td><?= htmlspecialchars($stage) ?></td>
                <td><?= htmlspecialchars($stageData['status'] ?? 'Pending') ?></td>
                <td>
                    <?php if($assigned): ?>
                        <details><summary>Show <?= count($assigned) ?> assigned</summary>
                            <ul style="margin:4px 0 0 16px;padding:0;list-style:disc;">
                                <?php foreach($assigned as $id): ?>
                                    <li><?= htmlspecialchars($staffList[$id] ?? 'Unknown') ?></li>
                                <?php endforeach;?>
                            </ul></details>
                    <?php else: ?>— none —<?php endif;?>

                    <?php if(!$locked):?>
                        <div class="checkbox-group" style="margin-top:8px;">
                            <?php foreach($staffList as $staffId=>$staffName): ?>
                                <label><input type="checkbox" name="assigned_to[]" value="<?= $staffId ?>" <?= in_array($staffId,$assigned)?'checked':''?>> <?= htmlspecialchars($staffName) ?></label>
                            <?php endforeach;?>
                        </div>
                    <?php endif;?>
                </td>
                <td><?= htmlspecialchars($stageData['started_at'] ?? '') ?></td>
                <td><?= htmlspecialchars($stageData['completed_at'] ?? '') ?></td>
                <td><input type="text" name="remarks" value="<?= htmlspecialchars($stageData['remarks'] ?? '') ?>" <?= $locked?'readonly':'' ?>></td>
                <td><?php if(!$locked):?><button type="submit" class="btn btn-primary">Save Assignment</button><?php else:?><button type="button" class="btn btn-disabled" disabled>Locked</button><?php endif;?></td>
            </form>
        </tr>
    <?php endforeach;?>
    </tbody>
</table>

<?php if ($allStagesCompleted && !$job_dispatched && $fullyDelivered): ?>

    <form method="post">
        <input type="hidden" name="dispatch" value="1">
        <button type="submit" class="btn btn-primary">Dispatch Job</button>
    </form>
<?php elseif (!$fullyDelivered): ?>
    <p class="message error"><em>Cannot dispatch: Delivery not complete. Items left to deliver: <?= $leftToDeliver ?></em></p>
<?php elseif ($job_dispatched): ?>
    <p><strong>Job dispatched on <?= htmlspecialchars($dispatched_at) ?></strong></p>

    <?php if ($deliveryNoteExists): ?>
        <button type="button" disabled>Upload Delivery Note</button>
        <a href="<?= htmlspecialchars($deliveryNotePath) ?>" target="_blank" style="margin-left: 10px;">View Delivery Note</a>
    <?php else: ?>
        <form method="post" action="upload_note.php" enctype="multipart/form-data">
            <label for="delivery_note_file">Upload Delivery Note (PDF):</label><br>
            <input type="file" name="note_file" id="delivery_note_file" accept="application/pdf" required>
            <input type="hidden" name="job_card_id" value="<?= $job_card_id ?>">
            <button type="submit">Upload Delivery Note</button>
        </form>
        <a href="generate_reports.php?job_card_id=<?= $job_card_id ?>" target="_blank"
           style="background:#de1a1c;color:white;padding:6px 12px;border-radius:6px;text-decoration:none;display:inline-block;margin-top:10px;">
            📄 Generate Report
        </a>
    <?php endif;?>
<?php endif;?>

</body>
</html>
