<?php
session_start();

// Redirect to login page if not logged in
if (!isset($_SESSION['username']) || !isset($_SESSION['branch'])) {
    header("Location: login.php");
    exit();
}

$branch = $_SESSION['branch'];
$role = $_SESSION['role'] ?? 'guest';
$username = $_SESSION['username'];
$is_admin = ($username === 'Admin');

// Database connection
$mysqli = new mysqli("localhost", "root", "", "mas_dashboard");
if ($mysqli->connect_errno) {
    die("Failed to connect: " . $mysqli->connect_error);
}

$today = date('Y-m-d');

// Due today by department for modal - filtered by branch
$dueTodayByDept = [];
$stmt = $mysqli->prepare("
    SELECT department, GROUP_CONCAT(job_card_number SEPARATOR ', ') AS job_cards, COUNT(*) AS total_due 
    FROM job_cards 
    WHERE status = 'open' AND due_date = ? AND branch = ? 
    GROUP BY department
");
$stmt->bind_param("ss", $today, $branch);
$stmt->execute();
$result = $stmt->get_result();
while ($row = $result->fetch_assoc()) {
    $dueTodayByDept[$row['department']] = [
        'job_cards' => $row['job_cards'],
        'total_due' => $row['total_due'],
    ];
}
$stmt->close();

// Job card stats filtered by branch
// Open job cards count
$stmt = $mysqli->prepare("SELECT COUNT(*) as total FROM job_cards WHERE status = 'open' AND branch = ?");
$stmt->bind_param("s", $branch);
$stmt->execute();
$open = $stmt->get_result()->fetch_assoc()['total'];
$stmt->close();

// Closed job cards count
$stmt = $mysqli->prepare("SELECT COUNT(*) as total FROM job_cards WHERE status = 'closed' AND branch = ?");
$stmt->bind_param("s", $branch);
$stmt->execute();
$closed = $stmt->get_result()->fetch_assoc()['total'];
$stmt->close();

$backlog = ($open + $closed) > 0 ? round(($open / ($open + $closed)) * 100) : 0;

// Due today count
$stmt = $mysqli->prepare("SELECT COUNT(*) as total FROM job_cards WHERE due_date = ? AND branch = ?");
$stmt->bind_param("ss", $today, $branch);
$stmt->execute();
$dueToday = $stmt->get_result()->fetch_assoc()['total'];
$stmt->close();

// Completed today count
$stmt = $mysqli->prepare("SELECT COUNT(*) as total FROM job_cards WHERE completed_date = ? AND branch = ?");
$stmt->bind_param("ss", $today, $branch);
$stmt->execute();
$completedToday = $stmt->get_result()->fetch_assoc()['total'];
$stmt->close();

$downtime = 52; // static value

$departments = ['Bulk Printing', 'Signage', 'Commercial Branding'];
$deliveries = [];

foreach ($departments as $dept) {
    // Get all orders for the department filtered by branch
    $ordersRes = $mysqli->prepare("SELECT id FROM job_cards WHERE department = ? AND branch = ?");
    $ordersRes->bind_param("ss", $dept, $branch);
    $ordersRes->execute();
    $ordersResult = $ordersRes->get_result();
    $orders = [];
    while ($row = $ordersResult->fetch_assoc()) {
        $orders[] = $row['id'];
    }
    $ordersRes->close();

    $total_orders = count($orders);
    $delivered = 0;
    $due = 0; // initialize

    if ($total_orders > 0) {
        $ids = implode(",", $orders);

        // Count delivered orders for these job_card_ids
        $deliveryRes = $mysqli->query("SELECT COUNT(DISTINCT job_card_id) as count FROM delivery_notes WHERE job_card_id IN ($ids)");
        $delivered = $deliveryRes->fetch_assoc()['count'];

        // Count due orders (status open and due_date <= today)
        $dueRes = $mysqli->prepare("
            SELECT COUNT(*) as due_count 
            FROM job_cards 
            WHERE department = ? 
              AND status = 'open' 
              AND due_date <= CURDATE()
              AND branch = ?
        ");
        $dueRes->bind_param("ss", $dept, $branch);
        $dueRes->execute();
        $due = $dueRes->get_result()->fetch_assoc()['due_count'];
        $dueRes->close();
    }

    $deliveries[] = [
        'department' => $dept,
        'orders' => $total_orders,
        'delivered' => $delivered,
        'due' => $due,
        'status' => $total_orders > 0 ? round(($delivered / $total_orders) * 100) : 0
    ];
}

// Due orders detail and returned cards filtered by branch
$dueOrdersDetail = [];
$returnedCards = [];

$query = "
    SELECT job_card_number, department, status, due_date, DATEDIFF(due_date, CURDATE()) as days_diff
    FROM job_cards
    WHERE status = 'open' AND status != 'Returned' AND branch = ?
";

$stmt = $mysqli->prepare($query);
$stmt->bind_param("s", $branch);
$stmt->execute();
$result = $stmt->get_result();

while ($row = $result->fetch_assoc()) {
    $dept = $row['department'];
    $status = $row['status'];
    $diff = (int)$row['days_diff'];
    $job_card_number = $row['job_card_number'];

    if (!isset($dueOrdersDetail[$dept])) {
        $dueOrdersDetail[$dept] = [];
    }

    if ($status === 'Returned') {
        $returnedCards[$dept][] = $job_card_number;
        continue;
    }

    if ($diff < 0) {
        $label = "Overdue by " . abs($diff) . " day" . (abs($diff) > 1 ? "s" : "");
    } elseif ($diff == 0) {
        $label = "Due today";
    } else {
        $label = "Due in " . $diff . " day" . ($diff > 1 ? "s" : "");
    }

    if (!isset($dueOrdersDetail[$dept][$label])) {
        $dueOrdersDetail[$dept][$label] = [];
    }

    $dueOrdersDetail[$dept][$label][] = $job_card_number;
}

$stmt->close();

// Employee hours filtered by branch
$employeeHoursStmt = $mysqli->prepare("
    SELECT employee_name, department, hours_worked
    FROM employee_hours
    WHERE branch = ?
");
$employeeHoursStmt->bind_param("s", $branch);
$employeeHoursStmt->execute();
$employeeHours = $employeeHoursStmt->get_result();

if (!$employeeHours) {
    die("Query failed: (" . $mysqli->errno . ") " . $mysqli->error);
}

if ($employeeHours->num_rows === 0) {
    // No hours logged at all
}

// Get Monthly Job Card Data per department, filtered by branch
$currentYear = date('Y');
$startMonth = date('n') - 3;
$endMonth = date('n');
if ($startMonth <= 0) {
    $startMonth += 12;
}

$months = [];
for ($i = $startMonth; $i <= $endMonth; $i++) {
    $monthNum = $i > 12 ? $i - 12 : $i;
    $months[$monthNum] = date('M', mktime(0, 0, 0, $monthNum, 1));
}

// Initialize the month data structure
$monthlyOrderData = [];
$departmentData = [];

foreach ($departments as $dept) {
    $departmentData[$dept] = [];
    foreach (range($startMonth, $endMonth) as $monthNum) {
        $actualMonth = $monthNum > 12 ? $monthNum - 12 : $monthNum;
        $departmentData[$dept][$actualMonth] = [
            'orders' => 0,
            'delivered' => 0
        ];
    }
}

// Use prepared statements for order and delivered queries with branch filter

$orderQuery = "
    SELECT 
        MONTH(created_at) as month,
        department,
        COUNT(*) as total_orders
    FROM 
        job_cards
    WHERE 
        YEAR(created_at) = ? AND MONTH(created_at) BETWEEN ? AND ? AND branch = ?
    GROUP BY 
        MONTH(created_at), department
";
$stmtOrder = $mysqli->prepare($orderQuery);
$stmtOrder->bind_param("iiis", $currentYear, $startMonth, $endMonth, $branch);
$stmtOrder->execute();
$orderResult = $stmtOrder->get_result();

$deliveredQuery = "
    SELECT 
        MONTH(jc.created_at) as month,
        jc.department,
        COUNT(DISTINCT dn.job_card_id) as delivered
    FROM 
        job_cards jc
    JOIN 
        delivery_notes dn ON jc.id = dn.job_card_id
    WHERE 
        YEAR(jc.created_at) = ? AND MONTH(jc.created_at) BETWEEN ? AND ? AND jc.branch = ?
    GROUP BY 
        MONTH(jc.created_at), jc.department
";
$stmtDelivered = $mysqli->prepare($deliveredQuery);
$stmtDelivered->bind_param("iiis", $currentYear, $startMonth, $endMonth, $branch);
$stmtDelivered->execute();
$deliveredResult = $stmtDelivered->get_result();

// Populate orders data by department
while ($row = $orderResult->fetch_assoc()) {
    $monthNum = (int)$row['month'];
    $dept = $row['department'];

    if (isset($departmentData[$dept][$monthNum])) {
        $departmentData[$dept][$monthNum]['orders'] = (int)$row['total_orders'];
    }
}

// Populate delivered data by department
while ($row = $deliveredResult->fetch_assoc()) {
    $monthNum = (int)$row['month'];
    $dept = $row['department'];

    if (isset($departmentData[$dept][$monthNum])) {
        $departmentData[$dept][$monthNum]['delivered'] = (int)$row['delivered'];
    }
}

$stmtOrder->close();
$stmtDelivered->close();

$dashboardLinks = [
    'sales' => 'insert.php',
    'accounts' => 'Accounts.php',
    'admin' => 'index.php',
    'Bulk Printing' => 'Admins.php',
    'Signage' => 'Admins.php',
    'Commercial Branding' => 'Admins.php',
];

// Determine dashboard button visibility
$showDashboardButton = array_key_exists($role, $dashboardLinks);
$dashboardUrl = $showDashboardButton ? $dashboardLinks[$role] : '#';

$canEdit = ($role === 'sales');
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <h2>Masteka Printing Production</h2>


    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" />
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

    <style>
        body {
            font-family: Arial, sans-serif;
            background: #eef6fc;
            margin: 0;
            padding: 20px;
        }
        h3 {
            text-align: center;
            font-size: 24px;
            color: #30307d;
            margin-bottom: 10px;
        }
        nav {
            margin-bottom: 20px;
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }
        nav a {
            background-color: #30307d;
            color: white;
            text-decoration: none;
            padding: 4px 16px;
            border-radius: 6px;
            font-weight: bold;
            transition: background-color 0.3s ease, transform 0.2s ease;
        }
        nav a:hover {
            background-color: #1f1f5c;
            transform: translateY(-2px);
        }
        .stats {
            display: flex;
            justify-content: space-between;
            flex-wrap: wrap;
        }
        .stat-card {
            flex: 1;
            min-width: 100px;
            margin: 10px;
            background: #30307d;
            color: white;
            padding: 4px;
            border-radius: 10px;
            text-align: center;
        }
        .dashboard {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 20px;
        }
        .card {
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .progress-bar {
            background: #e0e0e0;
            height: 12px;
            border-radius: 6px;
            margin: 6px 0;
        }
        .progress {
            background: #f36e25;
            height: 100%;
            border-radius: 6px;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 10px;
        }
        th, td {
            padding: 10px;
            text-align: center;
            border: 1px solid #ccc;
        }
        #gaugeChart {
            max-width: 200px;
            max-height: 200px;
            display: block;
            margin: 0 auto;
        }
        /* Department tabs styles */
        .tabs {
            display: flex;
            margin-bottom: 15px;
            border-bottom: 1px solid #ccc;
        }
        .tab {
            padding: 8px 16px;
            cursor: pointer;
            background: #eee;
            border: 1px solid #ccc;
            border-bottom: none;
            margin-right: 5px;
            border-radius: 5px 5px 0 0;
            font-weight: bold;
        }
        .tab.active {
            background: #30307d;
            color: white;
            border-color: #30307d;
        }
        .tab-content {
            display: none;
        }
        .tab-content.active {
            display: block;
        }
        .tab-buttons { list-style: none; display: flex; gap: 1rem; padding: 0; }
        .tab-button { background: #f36e25; color: #fff; border: none; padding: 0.5rem 1rem; cursor: pointer; border-radius: 6px; }
        .tab-button.active { background: #30307d; }
        .tab-content { display: none; margin-top: 1rem; }
        .tab-content.active { display: block; }
        canvas {
            height: 250px !important;
            width: 100% !important;
            max-height: 250px;
        }
    </style>
</head>
<body>
<?php if ($role === 'guest'): ?>
    <nav>
        <a href="login.php" style="background-color: #30307d; color: white; padding: 4px 16px; border-radius: 6px; font-weight: bold; text-decoration: none;">
            Login
        </a>
    </nav>

<?php elseif ($showDashboardButton): ?>

<?php endif; ?>


<h3><?= htmlspecialchars(ucfirst(strtolower($branch))) ?> Branch Dashboard</h3>


<?php if ($showDashboardButton): ?>
    <nav>
        <a href="<?= htmlspecialchars($dashboardUrl) ?>"><?= ucfirst($role) ?> Dashboard</a>
        <div style="margin-left: auto;">
            <a href="logout.php" style="background-color: #de1a1c;">Logout</a>
        </div>
    </nav>
<?php endif; ?>


<div class="stats">
    <div class="stat-card">Open Job Cards<br><strong><?= $open ?></strong></div>
    <div class="stat-card">Closed Job Cards<br><strong><?= $closed ?></strong></div>
    <div class="stat-card">Backlog<br><strong><?= $backlog ?>%</strong></div>
    <div class="stat-card" id="dueTodayCard" style="cursor:pointer; background-color: #de1a1c; color: white; font-weight: bold;">
        Due Orders
    </div>
    <div class="stat-card">Completed Today<br><strong><?= $completedToday ?></strong></div>


    <!-- Form  -->
    <form id="reportForm">
        <label for="report_type">Select Report Type:</label>
        <select name="type" id="report_type" required> <!-- change name to "type" -->
            <option value="job_cards">Job Cards</option>
            <option value="overdue">Overdue Jobs</option>
            <option value="delivery_performance">Delivery Performance</option>
            <option value="returned">Returned Jobs</option>
        </select>

        <label for="report_month">Select Month:</label>
        <input type="month" name="month" id="report_month" required> <!-- change name to "month" -->

        <button type="submit" class="btn btn-primary">Generate Report</button>
    </form>



    <!-- Modal -->
    <div id="reportModal" style="display:none;">
        <div style="position: fixed; top:0; left:0; width:100%; height:100%; background: rgba(0,0,0,0.7);">
            <div style="position:relative; max-width:90%; margin: 50px auto; background:white; padding:10px; border-radius:10px;">
                <span id="closeModal" style="position:absolute; top:10px; right:20px; font-size:24px; cursor:pointer;">&times;</span>
                <iframe id="reportFrame" src="" style="width:100%; height:80vh; border:none;"></iframe>
            </div>
        </div>
    </div>

</div>

<div class="dashboard">
    <div class="card delivery-updates">
        <h2>Delivery Updates </h2>
        <table>
            <tr>
                <th>Department</th>
                <th>Orders</th>
                <th>Delivered</th>
                <th>Pending</th>
                <th>Due</th> <!-- New column -->
                <th>Status</th>
            </tr>
            <?php foreach ($deliveries as $row): ?>
                <tr>
                    <td><?= htmlspecialchars($row['department']) ?></td>
                    <td><?= $row['orders'] ?></td>
                    <td><?= $row['delivered'] ?></td>
                    <td><?= max(0, $row['orders'] - $row['delivered']) ?></td>
                    <td><?= $row['due'] ?></td> <!-- Display due orders -->
                    <td><?= $row['status'] ?>%</td>
                </tr>
            <?php endforeach; ?>
        </table>
    </div>

    <div class="card performance-gauge">
        <h2>Returned Jobs per Department</h2>
        <table style="width:100%; border-collapse: collapse;">
            <thead>
            <tr style="background:#f2f2f2;">
                <th style="padding:8px; border:1px solid #ccc;">Original Job Card</th>
                <th style="padding:8px; border:1px solid #ccc;">New Job Card</th>
                <th style="padding:8px; border:1px solid #ccc;">Department</th>
                <th style="padding:8px; border:1px solid #ccc;">Reason</th>
                <th style="padding:8px; border:1px solid #ccc;">Date</th>
            </tr>
            </thead>
            <tbody>
            <?php
            $query = "
    SELECT r.*, j.department 
    FROM job_card_reissues r 
    LEFT JOIN job_cards j ON r.original_job_card_number = j.job_card_number 
    WHERE j.branch = ?
    ORDER BY r.created_at DESC
";

            $stmt = $mysqli->prepare($query);
            $stmt->bind_param("s", $branch);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($result && $result->num_rows > 0):
                while ($row = $result->fetch_assoc()):
                    ?>
                    <tr>
                        <td style="padding:8px; border:1px solid #ccc;"><?= htmlspecialchars($row['original_job_card_number']) ?></td>
                        <td style="padding:8px; border:1px solid #ccc;"><?= htmlspecialchars($row['new_job_card_number']) ?></td>
                        <td style="padding:8px; border:1px solid #ccc;"><?= htmlspecialchars($row['department']) ?></td>
                        <td style="padding:8px; border:1px solid #ccc;"><?= htmlspecialchars($row['reason']) ?></td>
                        <td style="padding:8px; border:1px solid #ccc;"><?= htmlspecialchars($row['created_at']) ?></td>
                    </tr>
                <?php
                endwhile;
            else:
                ?>
                <tr>
                    <td colspan="5" style="text-align:center;">No returned jobs found.</td>
                </tr>
            <?php
            endif;
            $stmt->close();
            ?>
            </tbody>

        </table>
    </div>


    <div class="card orders-chart">
        <h2>Orders vs Delivery Per Department </h2>

        <div class="tabs">
            <?php foreach ($departments as $index => $dept):
                $isActive = $index === 0 ? 'active' : '';
                $deptId = str_replace(' ', '', strtolower($dept));
                ?>
                <div class="tab <?= $isActive ?>" data-tab="<?= $deptId ?>"><?= $dept ?></div>
            <?php endforeach; ?>
        </div>

        <?php foreach ($departments as $index => $dept):
            $isActive = $index === 0 ? 'active' : '';
            $deptId = str_replace(' ', '', strtolower($dept));
            ?>
            <div class="tab-content <?= $isActive ?>" id="<?= $deptId ?>">
                <canvas id="chart-<?= $deptId ?>"></canvas>
            </div>
        <?php endforeach; ?>
    </div>

    <div class="card performance-gauge">
        <h2>Production Performance Gauge</h2>
        <canvas id="gaugeChart"></canvas>
    </div>

</div>

<script>
    // Tabs switching
    const tabs = document.querySelectorAll('.tab');
    const tabContents = document.querySelectorAll('.tab-content');

    tabs.forEach(tab => {
        tab.addEventListener('click', () => {
            tabs.forEach(t => t.classList.remove('active'));
            tab.classList.add('active');

            const target = tab.dataset.tab;
            tabContents.forEach(tc => {
                if (tc.id === target) {
                    tc.classList.add('active');
                } else {
                    tc.classList.remove('active');
                }
            });
        });
    });

    // Chart data from PHP
    const departmentData = <?= json_encode($departmentData) ?>;
    const months = <?= json_encode($months) ?>;

    // Setup charts for each department
    Object.keys(departmentData).forEach(dept => {
        const ctx = document.getElementById('chart-' + dept.replace(/\s+/g, '').toLowerCase()).getContext('2d');
        const data = departmentData[dept];

        const labels = Object.keys(data).map(m => months[m]);
        const orders = Object.values(data).map(d => d.orders);
        const delivered = Object.values(data).map(d => d.delivered);

        new Chart(ctx, {
            type: 'bar',
            data: {
                labels,
                datasets: [
                    {
                        label: 'Orders',
                        data: orders,
                        backgroundColor: 'rgba(243, 110, 37, 0.7)'
                    },
                    {
                        label: 'Delivered',
                        data: delivered,
                        backgroundColor: 'rgba(48, 48, 125, 0.7)'
                    }
                ]
            },
            options: {
                responsive: true,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: { stepSize: 1 }
                    }
                }
            }
        });
    });

    const ctxGauge = document.getElementById('gaugeChart').getContext('2d');
    const performance = <?= json_encode($backlog) ?>;

    const gaugeChart = new Chart(ctxGauge, {
        type: 'doughnut',
        data: {
            labels: ['Backlog', 'Completed'],
            datasets: [{
                data: [performance, 100 - performance],
                backgroundColor: ['#f36e25', '#ccc'],
                borderWidth: 0
            }]
        },
        options: {
            circumference: 180,
            rotation: 270,
            cutout: '80%',
            plugins: {
                legend: { display: false },
                tooltip: { enabled: false }
            }
        },
        plugins: [{
            id: 'centerText',
            beforeDraw(chart) {
                const {ctx, width, height} = chart;
                ctx.restore();

                // Adjust font size relative to chart height
                let fontSize = (height / 100).toFixed(2);
                ctx.font = `${fontSize}em Arial`;
                ctx.textBaseline = 'middle';

                const text = performance + '%';
                const textX = Math.round((width - ctx.measureText(text).width) / 2);
                const textY = height / 1.3;

                ctx.fillStyle = '#30307d';
                ctx.fillText(text, textX, textY);
                ctx.save();
            }
        }]
    });

    // Due Today Card click event example (optional)
    document.getElementById('dueTodayCard').addEventListener('click', () => {
        alert("Total Due Orders Today: <?= $dueToday ?>");
    });

    // Report Form
    document.getElementById('reportForm').addEventListener('submit', function (e) {
        e.preventDefault();
        const reportType = document.getElementById('report_type').value;
        const reportMonth = document.getElementById('report_month').value;

        // Adjust URL accordingly, example:
        const url = `generate_report.php?type=${encodeURIComponent(reportType)}&month=${encodeURIComponent(reportMonth)}`;
        document.getElementById('reportFrame').src = url;
        document.getElementById('reportModal').style.display = 'block';
    });

    document.getElementById('closeModal').addEventListener('click', () => {
        document.getElementById('reportModal').style.display = 'none';
    });


        document.getElementById('reportForm').addEventListener('submit', function (e) {
        e.preventDefault(); // prevent normal form submission

        const reportType = document.getElementById('report_type').value;
        const reportMonth = document.getElementById('report_month').value;

        if (!reportType || !reportMonth) {
        alert('Please select both report type and month.');
        return;
    }

        // Build URL for report.php with query parameters
            const url = `generate_report.php?type=${encodeURIComponent(reportType)}&month=${encodeURIComponent(reportMonth)}`;

        // Set the iframe source to the report URL to load PDF
        document.getElementById('reportFrame').src = url;

        // Show the modal
        document.getElementById('reportModal').style.display = 'block';
    });

        // Close modal button
        document.getElementById('closeModal').addEventListener('click', function () {
        document.getElementById('reportModal').style.display = 'none';
        // Clear iframe src to stop loading if needed
        document.getElementById('reportFrame').src = '';
    });
    document.getElementById('dueTodayCard').addEventListener('click', () => {
        document.getElementById('dueTodayModal').style.display = 'block';
    });

    document.getElementById('closeDueTodayModal').addEventListener('click', () => {
        document.getElementById('dueTodayModal').style.display = 'none';
    });

    window.addEventListener('click', function(e) {
        const modal = document.getElementById('dueTodayModal');
        if (e.target === modal) {
            modal.style.display = 'none';
        }
    });
</script>
</body>
</html>
