<?php

session_start();
if (!isset($_SESSION['username'])) {
    header("Location: login.php");
    exit;
}

$mysqli = new mysqli("localhost", "root", "", "mas_dashboard");

// Helper function to get current week's Monday
function getCurrentWeekMonday() {
    $today = new DateTime();
    $monday = clone $today;
    $monday->modify('last monday');
    if ($monday > $today) {
        $monday->modify('-7 days');
    }
    return $monday->format('Y-m-d');
}

// Helper function to get week end (Sunday)
function getCurrentWeekSunday() {
    $today = new DateTime();
    $sunday = clone $today;
    $sunday->modify('next sunday');
    if ($sunday < $today) {
        $sunday->modify('+7 days');
    }
    return $sunday->format('Y-m-d');
}

$currentWeekStart = getCurrentWeekMonday();
$currentWeekEnd = getCurrentWeekSunday();

// Get departments and employees from database
$departmentsQuery = "SELECT DISTINCT department FROM employees WHERE status = 'Active' ORDER BY department";
$departmentsResult = $mysqli->query($departmentsQuery);
$departments = [];
$department_employees = [];

if ($departmentsResult && $departmentsResult->num_rows > 0) {
    while ($dept = $departmentsResult->fetch_assoc()) {
        $deptName = $dept['department'];
        $departments[] = $deptName;

        // Get employees for this department
        $employeesQuery = "SELECT employee_name, position, is_supervisor FROM employees WHERE department = ? AND status = 'Active' ORDER BY is_supervisor DESC, employee_name";
        $stmt = $mysqli->prepare($employeesQuery);
        $stmt->bind_param("s", $deptName);
        $stmt->execute();
        $employeesResult = $stmt->get_result();

        $department_employees[$deptName] = [];
        while ($emp = $employeesResult->fetch_assoc()) {
            $department_employees[$deptName][] = $emp['employee_name'];
        }
        $stmt->close();
    }
}

// Handle adding new employee
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_employee'])) {
    $employee_name = trim($_POST['employee_name']);
    $department = $_POST['department'];
    $position = $_POST['position'];
    $is_supervisor = isset($_POST['is_supervisor']) ? 1 : 0;
    $email = trim($_POST['email']);
    $phone = trim($_POST['phone']);

    if (!empty($employee_name) && !empty($department) && !empty($position)) {
        $stmt = $mysqli->prepare("INSERT INTO employees (employee_name, department, position, is_supervisor, email, phone, hire_date, status) VALUES (?, ?, ?, ?, ?, ?, CURDATE(), 'Active')");
        $stmt->bind_param("sssiss", $employee_name, $department, $position, $is_supervisor, $email, $phone);
        if ($stmt->execute()) {
            echo "<script>alert('Employee added successfully!');</script>";
        } else {
            echo "<script>alert('Error adding employee: " . $mysqli->error . "');</script>";
        }
        $stmt->close();
    }
}

// Handle work hours logging
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['log_hours'])) {
    $employee_name = $_POST['employee_name'];
    $department = $_POST['department'];
    $hours_worked = floatval($_POST['hours_worked']);
    $work_date = $_POST['work_date'];
    $job_card_id = isset($_POST['job_card_id']) ? intval($_POST['job_card_id']) : null;
    $notes = $_POST['notes'] ?? '';

    $stmt = $mysqli->prepare("INSERT INTO employee_hours (employee_name, department, hours_worked, work_date, job_card_id, notes) VALUES (?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("ssdsis", $employee_name, $department, $hours_worked, $work_date, $job_card_id, $notes);
    $stmt->execute();
    $stmt->close();
}

// Handle job card assignment
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['assign_job_card'])) {
    $job_card_id = intval($_POST['job_card_id']);
    $employee_name = $_POST['employee_name'];
    $assigned_date = date('Y-m-d');

    // Check if assignment already exists
    $checkStmt = $mysqli->prepare("SELECT COUNT(*) FROM job_card_assignments WHERE job_card_id = ? AND employee_name = ?");
    $checkStmt->bind_param("is", $job_card_id, $employee_name);
    $checkStmt->execute();
    $checkStmt->bind_result($count);
    $checkStmt->fetch();
    $checkStmt->close();

    if ($count == 0) {
        $stmt = $mysqli->prepare("INSERT INTO job_card_assignments (job_card_id, employee_name, assigned_date) VALUES (?, ?, ?)");
        $stmt->bind_param("iss", $job_card_id, $employee_name, $assigned_date);
        $stmt->execute();
        $stmt->close();
    }
}

// Handle delivery note upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['delivery_note'])) {
    $job_card_id = intval($_POST['job_card_id']);
    $uploaded_by = $_SESSION['username'] ?? 'Unknown';
    $file = $_FILES['delivery_note'];
    $uploadDir = 'uploads/';
    $fileName = time() . '_' . basename($file['name']);
    $targetPath = $uploadDir . $fileName;
    $current_date = date('Y-m-d');

    if (!is_dir($uploadDir)) mkdir($uploadDir, 0777, true);

    $checkStmt = $mysqli->prepare("SELECT COUNT(*) FROM delivery_notes WHERE job_card_id = ?");
    $checkStmt->bind_param("i", $job_card_id);
    $checkStmt->execute();
    $checkStmt->bind_result($count);
    $checkStmt->fetch();
    $checkStmt->close();

    if ($count == 0 && move_uploaded_file($file['tmp_name'], $targetPath)) {
        $stmt = $mysqli->prepare("INSERT INTO delivery_notes (job_card_id, note_file, uploaded_by) VALUES (?, ?, ?)");
        $stmt->bind_param("iss", $job_card_id, $targetPath, $uploaded_by);
        $stmt->execute();
        $stmt->close();

        $updateJob = $mysqli->prepare("UPDATE job_cards SET status = 'Closed', completed_date = ? WHERE id = ?");
        $updateJob->bind_param("si", $current_date, $job_card_id);
        $updateJob->execute();
        $updateJob->close();
    }
}

// Create tables if they don't exist
$mysqli->query("CREATE TABLE IF NOT EXISTS employee_hours (
    id INT AUTO_INCREMENT PRIMARY KEY,
    employee_name VARCHAR(255) NOT NULL,
    department VARCHAR(100) NOT NULL,
    hours_worked DECIMAL(5,2) NOT NULL,
    work_date DATE NOT NULL,
    job_card_id INT NULL,
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
)");

$mysqli->query("CREATE TABLE IF NOT EXISTS job_card_assignments (
    id INT AUTO_INCREMENT PRIMARY KEY,
    job_card_id INT NOT NULL,
    employee_name VARCHAR(255) NOT NULL,
    assigned_date DATE NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY unique_assignment (job_card_id, employee_name)
)");

?>

<!DOCTYPE html>
<html>
<head>
    <title>Department Delivery Dashboard</title>
    <style>
        body { font-family: Arial; margin: 20px; background: #f9f9f9; }
        h1 { color: #30307d; position: relative; }

        .dashboard-btn {
            float: right;
            background-color: #30307d;
            color: white;
            padding: 6px 12px;
            border-radius: 5px;
            text-decoration: none;
            font-weight: bold;
        }
        .dashboard-btn:hover {
            background-color: #de1a1c;
        }

        /* Search Styles */
        .search-container {
            margin: 20px 0;
            padding: 15px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }

        .search-box {
            width: 100%;
            padding: 10px;
            border: 2px solid #ddd;
            border-radius: 5px;
            font-size: 16px;
        }

        .search-box:focus {
            border-color: #30307d;
            outline: none;
        }

        .tabs { display: flex; gap: 10px; margin-bottom: 20px; border-bottom: 2px solid #ddd; flex-wrap: wrap; }
        .tab { padding: 10px 15px; cursor: pointer; background: #f0f0f0; border-radius: 8px 8px 0 0; border: none; font-weight: 600; color: #444; font-size: 14px; }
        .tab:hover { background: #e6e6e6; color: #30307d; }
        .tab.active { background: #30307d; color: white; }

        .tab-content { display: none; animation: fadeIn 0.3s ease-in-out; }
        .tab-content.active { display: block; }

        @keyframes fadeIn { from { opacity: 0; } to { opacity: 1; } }

        .status { margin: 10px 0; font-weight: bold; color: #30307d; }
        .department-section { margin-bottom: 40px; }
        .section-title { color: #30307d; border-bottom: 2px solid #30307d; padding-bottom: 5px; margin-bottom: 15px; }

        .week-info {
            background: #e8f4fd;
            padding: 10px;
            border-radius: 5px;
            margin: 10px 0;
            font-weight: bold;
            color: #30307d;
        }

        table { width: 100%; border-collapse: collapse; margin-top: 15px; background: white; box-shadow: 0 2px 6px rgba(0,0,0,0.05); }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: center; font-size: 14px; }
        th { background-color: #30307d; color: white; }
        tr:nth-child(even) { background-color: #f9f9f9; }

        .employee-table th { background-color: #de1a1c; }
        .supervisor-row { background-color: #e8f4fd !important; font-weight: bold; }

        .hours-form, .add-employee-form {
            background: #f0f8ff;
            padding: 15px;
            border-radius: 8px;
            margin: 15px 0;
        }
        .hours-form h4, .add-employee-form h4 { color: #30307d; margin-top: 0; }
        .form-row { display: flex; gap: 10px; margin-bottom: 10px; flex-wrap: wrap; }
        .form-group { flex: 1; min-width: 200px; }
        .form-group label { display: block; margin-bottom: 5px; font-weight: bold; color: #333; }
        .form-group select, .form-group input, .form-group textarea { width: 100%; padding: 8px; border: 1px solid #ccc; border-radius: 4px; }

        form { display: inline; }
        input[type="file"] { padding: 5px; }
        button { padding: 5px 10px; background: #30307d; color: white; border: none; border-radius: 4px; cursor: pointer; }
        button:hover { background: #1e1e5c; }
        .btn-hours { background: #de1a1c; }
        .btn-hours:hover { background: #b81619; }
        .btn-add { background: #28a745; }
        .btn-add:hover { background: #218838; }

        a { color: #f36e25; font-weight: bold; text-decoration: none; }
        a:hover { text-decoration: underline; }

        .summary-card { background: white; padding: 15px; border-radius: 8px; margin: 10px 0; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .summary-stats { display: flex; justify-content: space-between; flex-wrap: wrap; }
        .stat-item { text-align: center; padding: 10px; }
        .stat-number { font-size: 24px; font-weight: bold; color: #30307d; }
        .stat-label { font-size: 12px; color: #666; }

        .employee-info { background: #f8f9fa; padding: 10px; margin: 10px 0; border-radius: 5px; }
        .employee-count { color: #666; font-size: 14px; margin-bottom: 10px; }

        .assignment-form {
            background: #fff3cd;
            padding: 10px;
            border-radius: 5px;
            margin: 10px 0;
        }

        .job-assignments {
            background: #d4edda;
            padding: 5px;
            border-radius: 3px;
            font-size: 12px;
        }

        .hidden { display: none; }
    </style>
</head>
<body>

<h3>
    Department Delivery Dashboard
    <a href="index.php" class="dashboard-btn">Go to Dashboard</a>
</h3>

<!-- Week Information -->
<div class="week-info">
    Current Week: <?= date('M j', strtotime($currentWeekStart)) ?> - <?= date('M j, Y', strtotime($currentWeekEnd)) ?>
    (Week resets every Monday)
</div>

<!-- Global Search -->
<div class="search-container">
    <input type="text" id="globalSearch" class="search-box" placeholder="Search employees, job cards, departments...">
</div>

<div class="tabs">
    <?php foreach ($departments as $index => $dept): ?>
        <button class="tab<?= $index === 0 ? ' active' : '' ?>" onclick="showTab(<?= $index ?>)"><?= htmlspecialchars($dept) ?></button>
    <?php endforeach; ?>
    <button class="tab" onclick="showTab('history')">Upload History</button>
    <button class="tab" onclick="showTab('hours-summary')">Hours Summary</button>
    <button class="tab" onclick="showTab('employee-directory')">Employee Directory</button>
    <button class="tab" onclick="showTab('add-staff')">Add Staff</button>
</div>

<?php foreach ($departments as $index => $dept): ?>
<div class="tab-content<?= $index === 0 ? ' active' : '' ?>" id="tab-<?= $index ?>">

    <!-- Job Cards Section -->
    <div class="department-section">
        <h3 class="section-title"><?= htmlspecialchars($dept) ?> - Job Cards & Assignments</h3>
        <?php
        $ordersRes = $mysqli->query("SELECT * FROM job_cards WHERE department = '" . $mysqli->real_escape_string($dept) . "'");
        $orders = $ordersRes ? $ordersRes->fetch_all(MYSQLI_ASSOC) : [];

        $notesRes = $mysqli->query("SELECT job_card_id FROM delivery_notes");
        $delivered_ids = [];
        if ($notesRes) {
            while ($row = $notesRes->fetch_assoc()) {
                $delivered_ids[] = $row['job_card_id'];
            }
        }

        $total_orders = count($orders);
        $delivered = 0;
        foreach ($orders as $o) {
            if (in_array($o['id'], $delivered_ids)) $delivered++;
        }
        $pending = $total_orders - $delivered;
        $percent = $total_orders > 0 ? round(($delivered / $total_orders) * 100) : 0;
        ?>

        <div class="status">
            Orders: <?= $total_orders ?> |
            Delivered: <?= $delivered ?> |
            Pending: <?= $pending ?> |
            Status: <?= ($pending === 0) ? "Closed" : "Open" ?> |
            Progress: <?= $percent ?>%
        </div>

        <!-- Job Card Assignment Form -->
        <?php if (!empty($orders)): ?>
            <div class="assignment-form">
                <h4>Assign Job Card to Employee</h4>
                <form method="POST">
                    <input type="hidden" name="assign_job_card" value="1">
                    <div class="form-row">
                        <div class="form-group">
                            <label>Job Card:</label>
                            <select name="job_card_id" required>
                                <option value="">Select Job Card</option>
                                <?php foreach ($orders as $order): ?>
                                    <option value="<?= $order['id'] ?>"><?= htmlspecialchars($order['job_card_number']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Employee:</label>
                            <select name="employee_name" required>
                                <option value="">Select Employee</option>
                                <?php
                                $deptEmployeesQuery = "SELECT employee_name FROM employees WHERE department = ? AND status = 'Active' ORDER BY employee_name";
                                $stmt = $mysqli->prepare($deptEmployeesQuery);
                                $stmt->bind_param("s", $dept);
                                $stmt->execute();
                                $deptEmployeesResult = $stmt->get_result();
                                while ($emp = $deptEmployeesResult->fetch_assoc()):
                                    ?>
                                    <option value="<?= htmlspecialchars($emp['employee_name']) ?>"><?= htmlspecialchars($emp['employee_name']) ?></option>
                                <?php endwhile;
                                $stmt->close();
                                ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <button type="submit" class="btn-add">Assign Job Card</button>
                        </div>
                    </div>
                </form>
            </div>
        <?php endif; ?>

        <?php if (!empty($orders)): ?>
            <table class="searchable">
                <thead>
                <tr>
                    <th>Job Card #</th>
                    <th>Due Date</th>
                    <th>Assigned To</th>
                    <th>Completed Date</th>
                    <th>Delivery Note</th>
                    <th>Status</th>
                    <th>Uploaded By</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($orders as $order): ?>
                    <?php
                    $job_card_id = $order['id'];
                    $has_note = in_array($job_card_id, $delivered_ids);
                    $uploader = "";
                    $completed_date = "";

                    // Get assignments
                    $assignmentRes = $mysqli->query("SELECT employee_name FROM job_card_assignments WHERE job_card_id = $job_card_id");
                    $assignments = [];
                    if ($assignmentRes) {
                        while ($assign = $assignmentRes->fetch_assoc()) {
                            $assignments[] = $assign['employee_name'];
                        }
                    }

                    if ($has_note) {
                        $noteResult = $mysqli->query("SELECT note_file, uploaded_by FROM delivery_notes WHERE job_card_id = $job_card_id LIMIT 1");
                        if ($noteResult) {
                            $note = $noteResult->fetch_assoc();
                            $uploader = $note['uploaded_by'];
                        }

                        $completedResult = $mysqli->query("SELECT completed_date FROM job_cards WHERE id = $job_card_id LIMIT 1");
                        if ($completedResult) {
                            $completed = $completedResult->fetch_assoc();
                            $completed_date = $completed['completed_date'] ?? "-";
                        }
                    }
                    ?>
                    <tr>
                        <td><?= htmlspecialchars($order['job_card_number']) ?></td>
                        <td><?= htmlspecialchars($order['due_date']) ?></td>
                        <td>
                            <?php if (!empty($assignments)): ?>
                                <div class="job-assignments">
                                    <?= implode(', ', array_map('htmlspecialchars', $assignments)) ?>
                                </div>
                            <?php else: ?>
                                <em>Not assigned</em>
                            <?php endif; ?>
                        </td>
                        <td><?= $has_note ? htmlspecialchars($completed_date) : "-" ?></td>
                        <td>
                            <?php if ($has_note): ?>
                                <a href="<?= htmlspecialchars($note['note_file']) ?>" target="_blank">📄 View Note</a>
                            <?php else: ?>
                                <form method="POST" enctype="multipart/form-data">
                                    <input type="hidden" name="job_card_id" value="<?= $job_card_id ?>">
                                    <input type="file" name="delivery_note" required>
                                    <button type="submit">Upload</button>
                                </form>
                            <?php endif; ?>
                        </td>
                        <td><?= $has_note ? "Delivered" : "Pending" ?></td>
                        <td><?= $has_note ? htmlspecialchars($uploader) : "-" ?></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        <?php else: ?>
            <p>No job cards found for this department.</p>
        <?php endif; ?>
    </div>

    <!-- Employees Section -->
    <div class="department-section">
        <h3 class="section-title"><?= htmlspecialchars($dept) ?> - Employees & Weekly Hours</h3>

        <?php
        // Get department employee info from database
        $deptEmployeesQuery = "SELECT employee_name, position, is_supervisor FROM employees WHERE department = ? AND status = 'Active' ORDER BY is_supervisor DESC, employee_name";
        $stmt = $mysqli->prepare($deptEmployeesQuery);
        $stmt->bind_param("s", $dept);
        $stmt->execute();
        $deptEmployeesResult = $stmt->get_result();
        $deptEmployees = $deptEmployeesResult->fetch_all(MYSQLI_ASSOC);
        $stmt->close();

        $totalEmployees = count($deptEmployees);
        $supervisors = array_filter($deptEmployees, function($emp) { return $emp['is_supervisor'] == 1; });
        $staff = array_filter($deptEmployees, function($emp) { return $emp['is_supervisor'] == 0; });
        ?>

        <div class="employee-count">
            Total Employees: <?= $totalEmployees ?> | Supervisors: <?= count($supervisors) ?> | Staff: <?= count($staff) ?>
        </div>

        <!-- Work Hours Logging Form -->
        <?php if (!empty($deptEmployees)): ?>
            <div class="hours-form">
                <h4>Log Work Hours</h4>
                <form method="POST">
                    <input type="hidden" name="log_hours" value="1">
                    <input type="hidden" name="department" value="<?= htmlspecialchars($dept) ?>">

                    <div class="form-row">
                        <div class="form-group">
                            <label>Employee:</label>
                            <select name="employee_name" required>
                                <option value="">Select Employee</option>
                                <?php foreach ($deptEmployees as $employee): ?>
                                    <option value="<?= htmlspecialchars($employee['employee_name']) ?>">
                                        <?= htmlspecialchars($employee['employee_name']) ?>
                                        (<?= $employee['is_supervisor'] ? 'Supervisor' : 'Staff' ?>)
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Hours Worked:</label>
                            <input type="number" name="hours_worked" step="0.25" min="0" max="35" required>
                        </div>
                        <div class="form-group">
                            <label>Work Date:</label>
                            <input type="date" name="work_date" value="<?= date('Y-m-d') ?>" required>
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label>Job Card (Optional):</label>
                            <select name="job_card_id">
                                <option value="">Select Job Card</option>
                                <?php foreach ($orders as $order): ?>
                                    <option value="<?= $order['id'] ?>"><?= htmlspecialchars($order['job_card_number']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Notes:</label>
                            <textarea name="notes" rows="2" placeholder="Optional work notes..."></textarea>
                        </div>
                    </div>

                    <button type="submit" class="btn-hours">Log Hours</button>
                </form>
            </div>
        <?php endif; ?>

        <!-- Employee Weekly Hours Summary -->
        <?php
        $hoursRes = $mysqli->query("
                SELECT 
                    eh.employee_name, 
                    SUM(eh.hours_worked) as weekly_hours,
                    COUNT(DISTINCT eh.work_date) as days_worked,
                    GROUP_CONCAT(DISTINCT jc.job_card_number ORDER BY jc.job_card_number SEPARATOR ', ') as assigned_job_cards
                FROM employee_hours eh
                LEFT JOIN job_card_assignments jca ON eh.employee_name = jca.employee_name
                LEFT JOIN job_cards jc ON jca.job_card_id = jc.id
                WHERE eh.department = '" . $mysqli->real_escape_string($dept) . "'
                AND eh.work_date >= '$currentWeekStart' 
                AND eh.work_date <= '$currentWeekEnd'
                GROUP BY eh.employee_name
                ORDER BY weekly_hours DESC
            ");
        $employeeWeeklyHours = $hoursRes ? $hoursRes->fetch_all(MYSQLI_ASSOC) : [];
        ?>

        <?php if (!empty($deptEmployees)): ?>
        <table class="employee-table searchable">
            <thead>
            <tr>
                <th>Employee Name</th>
                <th>Position</th>
                <th>Assigned Job Cards</th>
                <th>Weekly Hours</th>
                <th>Days Worked</th>
                <th>Avg Hours/Day</th>
                <th>Status</th>
            </tr>
            </thead>
            <tbody>
            <?php
            $employeeHoursMap = [];
            foreach ($employeeWeeklyHours as $emp) {
                $employeeHoursMap[$emp['employee_name']] = $emp;
            }

            foreach ($deptEmployees as $employee):
            $empData = $employeeHoursMap[$employee['employee_name']] ?? null;
            $weeklyHours = $empData ? $empData['weekly_hours'] : 0;
            $daysWorked = $empData ? $empData['days_worked'] : 0;
            $avgHours = $daysWorked > 0 ? round($weeklyHours / $daysWorked, 2) : 0;
            $assignedJobCards = $empData ? $empData['assigned_job_cards'] : '';
            $status = $weeklyHours > 0 ? 'Active This Week' : 'No Hours This Week';
            $isSupervisor = $employee['is_supervisor'] == 1;

            // Get all assigned job cards (not just those with hours)
            if (empty($assignedJobCards)) {
                $assignmentQuery = $mysqli->query("
                                SELECT GROUP_CONCAT(jc.job_card_number ORDER BY jc.job_card_number SEPARATOR ', ') as job_cards
                                FROM job_card_assignments jca
                                JOIN job_cards jc ON jca.job_card_id = jc.id
                                WHERE jca.employee_name = '" . $mysqli->real_escape_string($employee['employee_name']) . "'
                            ");
                if ($assignmentQuery) {
                    $assignmentResult = $assignmentQuery->fetch_assoc();
                    $assignedJobCards = $assignmentResult['job_cards'] ?? '';
                }
            }
            ?>
            <tr<?= $isSupervisor ? ' class="supervisor-row"' : '' ?>>
                <td><?= htmlspecialchars($employee['employee_name']) ?></td>
                <td><?= htmlspecialchars($employee['position']) ?><?= $isSupervisor ? ' ⭐' : '' ?></td>
                <td>
                    <?php if (!empty($assignedJobCards)): ?>
                        <div class="job-assignments"><?= htmlspecialchars($assignedJobCards) ?></div>
                    <?php else: ?>
                        <em>No assignments</em>
                    <?php endif; ?>
                </td>
                <td><?= number_format($weeklyHours, 2) ?></td>
                <td><?= $daysWorked ?></td>
                <td><?= number_format($avgHours, 2) ?></td>
                <td><?= $status ?></td>
            </tr>
            <?php endforeach; ?>
            </tbody>
        </table>
        <?php else: ?>
            <p>No employees found for this department.</p>
        <?php endif; ?>
    </div>
</div>
<?php endforeach; ?>

<!-- Upload History Tab -->
<div class="tab-content" id="tab-history">
    <div class="department-section">
        <h3 class="section-title">Delivery Note Upload History</h3>
        <?php
        $historyRes = $mysqli->query("
            SELECT dn.*, jc.job_card_number, jc.department 
            FROM delivery_notes dn 
            JOIN job_cards jc ON dn.job_card_id = jc.id 
            ORDER BY dn.created_at DESC
        ");
        $history = $historyRes ? $historyRes->fetch_all(MYSQLI_ASSOC) : [];
        ?>

        <?php if (!empty($history)): ?>
            <table class="searchable">
                <thead>
                <tr>
                    <th>Job Card #</th>
                    <th>Department</th>
                    <th>Upload Date</th>
                    <th>Uploaded By</th>
                    <th>File</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($history as $record): ?>
                    <tr>
                        <td><?= htmlspecialchars($record['job_card_number']) ?></td>
                        <td><?= htmlspecialchars($record['department']) ?></td>
                        <td><?= date('M j, Y g:i A', strtotime($record['created_at'])) ?></td>
                        <td><?= htmlspecialchars($record['uploaded_by']) ?></td>
                        <td><a href="<?= htmlspecialchars($record['note_file']) ?>" target="_blank">📄 View File</a></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        <?php else: ?>
            <p>No delivery notes have been uploaded yet.</p>
        <?php endif; ?>
    </div>
</div>

<!-- Hours Summary Tab -->
<div class="tab-content" id="tab-hours-summary">
    <div class="department-section">
        <h3 class="section-title">Weekly Hours Summary</h3>

        <?php
        $totalHoursRes = $mysqli->query("
            SELECT 
                department,
                COUNT(DISTINCT employee_name) as employee_count,
                SUM(hours_worked) as total_hours,
                AVG(hours_worked) as avg_hours_per_entry,
                COUNT(*) as total_entries
            FROM employee_hours 
            WHERE work_date >= '$currentWeekStart' AND work_date <= '$currentWeekEnd'
            GROUP BY department
            ORDER BY total_hours DESC
        ");
        $deptSummary = $totalHoursRes ? $totalHoursRes->fetch_all(MYSQLI_ASSOC) : [];
        ?>

        <div class="summary-card">
            <div class="summary-stats">
                <?php
                $totalWeekHours = array_sum(array_column($deptSummary, 'total_hours'));
                $totalActiveEmployees = array_sum(array_column($deptSummary, 'employee_count'));
                $avgHoursPerEmployee = $totalActiveEmployees > 0 ? round($totalWeekHours / $totalActiveEmployees, 2) : 0;
                ?>
                <div class="stat-item">
                    <div class="stat-number"><?= number_format($totalWeekHours, 1) ?></div>
                    <div class="stat-label">Total Hours This Week</div>
                </div>
                <div class="stat-item">
                    <div class="stat-number"><?= $totalActiveEmployees ?></div>
                    <div class="stat-label">Active Employees</div>
                </div>
                <div class="stat-item">
                    <div class="stat-number"><?= number_format($avgHoursPerEmployee, 1) ?></div>
                    <div class="stat-label">Avg Hours/Employee</div>
                </div>
            </div>
        </div>

        <?php if (!empty($deptSummary)): ?>
            <table class="searchable">
                <thead>
                <tr>
                    <th>Department</th>
                    <th>Active Employees</th>
                    <th>Total Hours</th>
                    <th>Avg Hours/Employee</th>
                    <th>Total Entries</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($deptSummary as $dept): ?>
                    <tr>
                        <td><?= htmlspecialchars($dept['department']) ?></td>
                        <td><?= $dept['employee_count'] ?></td>
                        <td><?= number_format($dept['total_hours'], 2) ?></td>
                        <td><?= number_format($dept['total_hours'] / $dept['employee_count'], 2) ?></td>
                        <td><?= $dept['total_entries'] ?></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        <?php else: ?>
            <p>No hours logged for this week yet.</p>
        <?php endif; ?>
    </div>
</div>

<!-- Employee Directory Tab -->
<div class="tab-content" id="tab-employee-directory">
    <div class="department-section">
        <h3 class="section-title">Employee Directory</h3>
        <?php
        $allEmployeesRes = $mysqli->query("
            SELECT e.*, 
                   COALESCE(SUM(eh.hours_worked), 0) as total_hours_this_week,
                   COUNT(DISTINCT eh.work_date) as days_worked_this_week
            FROM employees e
            LEFT JOIN employee_hours eh ON e.employee_name = eh.employee_name 
                AND eh.work_date >= '$currentWeekStart' 
                AND eh.work_date <= '$currentWeekEnd'
            WHERE e.status = 'Active'
            GROUP BY e.id
            ORDER BY e.department, e.is_supervisor DESC, e.employee_name
        ");
        $allEmployees = $allEmployeesRes ? $allEmployeesRes->fetch_all(MYSQLI_ASSOC) : [];
        ?>

        <?php if (!empty($allEmployees)): ?>
            <table class="employee-table searchable">
                <thead>
                <tr>
                    <th>Name</th>
                    <th>Department</th>
                    <th>Position</th>
                    <th>Email</th>
                    <th>Phone</th>
                    <th>Hire Date</th>
                    <th>Weekly Hours</th>
                    <th>Days Worked</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($allEmployees as $emp): ?>
                    <tr<?= $emp['is_supervisor'] ? ' class="supervisor-row"' : '' ?>>
                        <td><?= htmlspecialchars($emp['employee_name']) ?><?= $emp['is_supervisor'] ? ' ⭐' : '' ?></td>
                        <td><?= htmlspecialchars($emp['department']) ?></td>
                        <td><?= htmlspecialchars($emp['position']) ?></td>
                        <td><?= htmlspecialchars($emp['email'] ?? '-') ?></td>
                        <td><?= htmlspecialchars($emp['phone'] ?? '-') ?></td>
                        <td><?= $emp['hire_date'] ? date('M j, Y', strtotime($emp['hire_date'])) : '-' ?></td>
                        <td><?= number_format($emp['total_hours_this_week'], 2) ?></td>
                        <td><?= $emp['days_worked_this_week'] ?></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        <?php else: ?>
            <p>No employees found.</p>
        <?php endif; ?>
    </div>
</div>

<!-- Add Staff Tab -->
<div class="tab-content" id="tab-add-staff">
    <div class="department-section">
        <h3 class="section-title">Add New Employee</h3>

        <div class="add-employee-form">
            <form method="POST">
                <input type="hidden" name="add_employee" value="1">

                <div class="form-row">
                    <div class="form-group">
                        <label>Employee Name *:</label>
                        <input type="text" name="employee_name" required>
                    </div>
                    <div class="form-group">
                        <label>Department *:</label>
                        <select name="department" required>
                            <option value="">Select Department</option>
                            <?php foreach ($departments as $dept): ?>
                                <option value="<?= htmlspecialchars($dept) ?>"><?= htmlspecialchars($dept) ?></option>
                            <?php endforeach; ?>
                            <option value="NEW">Add New Department</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label>Position *:</label>
                        <input type="text" name="position" required>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Email:</label>
                        <input type="email" name="email">
                    </div>
                    <div class="form-group">
                        <label>Phone:</label>
                        <input type="tel" name="phone">
                    </div>
                    <div class="form-group">
                        <label>
                            <input type="checkbox" name="is_supervisor"> Is Supervisor
                        </label>
                    </div>
                </div>

                <button type="submit" class="btn-add">Add Employee</button>
            </form>
        </div>
    </div>
</div>

<script>
    function showTab(tabIndex) {
        // Hide all tab contents
        const contents = document.querySelectorAll('.tab-content');
        contents.forEach(content => content.classList.remove('active'));

        // Remove active class from all tabs
        const tabs = document.querySelectorAll('.tab');
        tabs.forEach(tab => tab.classList.remove('active'));

        // Show selected tab content
        if (typeof tabIndex === 'string') {
            document.getElementById('tab-' + tabIndex).classList.add('active');
            // Find and activate the corresponding tab button
            const tabButtons = document.querySelectorAll('.tab');
            tabButtons.forEach((tab, index) => {
                if (tab.textContent.toLowerCase().includes(tabIndex.replace('-', ' '))) {
                    tab.classList.add('active');
                }
            });
        } else {
            document.getElementById('tab-' + tabIndex).classList.add('active');
            document.querySelectorAll('.tab')[tabIndex].classList.add('active');
        }
    }

    // Global search functionality
    document.getElementById('globalSearch').addEventListener('input', function() {
        const searchTerm = this.value.toLowerCase();
        const tables = document.querySelectorAll('.searchable');

        tables.forEach(table => {
            const rows = table.querySelectorAll('tbody tr');
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                if (text.includes(searchTerm)) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        });
    });

    // Initialize first tab as active
    document.addEventListener('DOMContentLoaded', function() {
        showTab(0);
    });
</script>

</body>
</html>