<?php
session_start();
$mysqli = new mysqli("localhost", "root", "", "mas_dashboard");
if ($mysqli->connect_errno) {
    die("Failed to connect to MySQL: " . $mysqli->connect_error);
}

$message = '';

$departments = [];
$res = $mysqli->query("SELECT DISTINCT department FROM job_cards WHERE department IS NOT NULL ORDER BY department ASC");
while ($d = $res->fetch_assoc()) {
    $departments[] = $d['department'];
}
array_unshift($departments, 'All');

$selected_department = $_GET['department'] ?? 'All';
$search = trim($_GET['search'] ?? '');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $id = (int)($_POST['job_card_id'] ?? 0);
    $invoice_number = trim($_POST['invoice_number'] ?? '');

    if ($id && $invoice_number !== '') {
        $stmt = $mysqli->prepare("UPDATE job_cards SET invoice_number = ?, invoiced = 1, status = 'dispatched' WHERE id = ? AND dispatched = 1 AND invoiced = 0");
        $stmt->bind_param('si', $invoice_number, $id);
        $stmt->execute();

        if ($stmt->affected_rows > 0) {
            $stmt2 = $mysqli->prepare("SELECT job_card_number FROM job_cards WHERE id = ?");
            $stmt2->bind_param('i', $id);
            $stmt2->execute();
            $stmt2->bind_result($job_card_number);
            $stmt2->fetch();
            $stmt2->close();

            $message = "Job Card #$job_card_number invoiced successfully.";
        } else {
            $message = "Failed to invoice. Ensure the job is dispatched and not already invoiced.";
        }
        $stmt->close();
    } else {
        $message = "Invoice number is required.";
    }
}

$sql = "SELECT * FROM job_cards";
$params = [];
$types = '';
$whereClauses = [];
if ($selected_department !== 'All') {
    $whereClauses[] = "department = ?";
    $types .= 's';
    $params[] = $selected_department;
}
if ($search !== '') {
    $whereClauses[] = "(job_card_number LIKE ? OR customer_name LIKE ?)";
    $types .= 'ss';
    $searchLike = "%$search%";
    $params[] = $searchLike;
    $params[] = $searchLike;
}
if (count($whereClauses) > 0) {
    $sql .= " WHERE " . implode(' AND ', $whereClauses);
}
$sql .= " ORDER BY due_date ASC";

$stmt = $mysqli->prepare($sql);
if ($types) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$result = $stmt->get_result();

$job_card_ids = [];
$job_cards = [];
while ($job = $result->fetch_assoc()) {
    $job_card_ids[] = $job['id'];
    $job_cards[$job['id']] = $job;
}

// Fetch stages per job card
if (count($job_card_ids) > 0) {
    $placeholders = implode(',', array_fill(0, count($job_card_ids), '?'));
    $stage_sql = "
      SELECT job_card_id, GROUP_CONCAT(stage_name ORDER BY started_at SEPARATOR ', ') AS stages
      FROM job_card_stages
      WHERE job_card_id IN ($placeholders)
      GROUP BY job_card_id
    ";
    $stage_stmt = $mysqli->prepare($stage_sql);
    $types_stages = str_repeat('i', count($job_card_ids));
    $stage_stmt->bind_param($types_stages, ...$job_card_ids);
    $stage_stmt->execute();
    $stage_res = $stage_stmt->get_result();

    while ($stage_row = $stage_res->fetch_assoc()) {
        $job_id = $stage_row['job_card_id'];
        $job_cards[$job_id]['stages'] = $stage_row['stages'];
    }
}

$countSql = "SELECT 
    COUNT(*) AS total,
    SUM(dispatched=1) AS dispatched_count,
    SUM(invoiced=1) AS invoiced_count,
    SUM(status='open') AS open_count,
    SUM(status='closed') AS closed_count,
    SUM(status='returned') AS returned_count,
    SUM((due_date <= CURDATE()) AND (status != 'closed')) AS due_count
    FROM job_cards";

if (count($whereClauses) > 0) {
    $countSql .= " WHERE " . implode(' AND ', $whereClauses);
}
$countStmt = $mysqli->prepare($countSql);
if ($types) {
    $countStmt->bind_param($types, ...$params);
}
$countStmt->execute();
$countRes = $countStmt->get_result()->fetch_assoc();

function statusClass($job) {
    $due_date = $job['due_date'] ?? null;
    $today = date('Y-m-d');
    $status = strtolower($job['status'] ?? '');
    $is_closed = $status === 'closed';
    $is_dispatched = (bool)$job['dispatched'];
    $is_due = false;

    if ($due_date && !$is_closed) {
        $is_due = ($due_date <= $today);
    }

    if ($is_dispatched && $is_closed) {
        return 'status-dispatched-closed';
    } elseif ($is_due) {
        return 'status-due';
    } elseif ($is_closed) {
        return 'status-closed';
    } else {
        return 'status-open';
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Admin - Job Cards & Accounts</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            background: #eef6fc;
            color: #30307d;
            padding: 20px;
        }
        h2 {
            margin-bottom: 10px;
        }
        .tabs {
            margin-bottom: 20px;
            display: flex;
            gap: 8px;
        }
        .tab {
            padding: 8px 16px;
            background: #d1d4e0;
            color: #30307d;
            text-decoration: none;
            border-radius: 5px;
            font-weight: bold;
            user-select: none;
        }
        .tab.active {
            background: #30307d;
            color: white;
        }
        .search-bar {
            margin-bottom: 15px;
        }
        input[type="search"] {
            padding: 6px 10px;
            font-size: 1em;
            border-radius: 5px;
            border: 1px solid #ccc;
            width: 280px;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
            font-size: 0.95em;
        }
        th, td {
            padding: 10px;
            border: 1px solid #ccc;
            text-align: left;
            vertical-align: middle;
        }
        th {
            background-color: #30307d;
            color: white;
        }
        .message {
            background: #d4edda;
            border: 1px solid #c3e6cb;
            padding: 10px;
            margin-bottom: 20px;
            color: #155724;
            border-radius: 5px;
            max-width: 600px;
        }
        .status-due {
            background: #f8d7da;
            color: #721c24;
        }
        .status-closed {
            background: #ffe5b4;
            color: #8a4b00;
        }
        .status-open {
            background: #d4edda;
            color: #155724;
        }
        .status-dispatched-closed {
            background: #c3e6cb;
            color: #155724;
        }
    </style>
</head>
<body>

<h2>Admin - Job Cards & Accounts</h2>

<!-- Department Tabs -->
<div class="tabs" role="tablist">
    <?php foreach ($departments as $dept): ?>
        <a href="?department=<?= urlencode($dept) ?>&search=<?= urlencode($search) ?>"
           class="tab <?= $selected_department === $dept ? 'active' : '' ?>"
           role="tab"><?= htmlspecialchars($dept) ?></a>
    <?php endforeach; ?>
</div>

<!-- Search Form -->
<form method="get" class="search-bar" role="search" aria-label="Search job cards">
    <input type="hidden" name="department" value="<?= htmlspecialchars($selected_department) ?>" />
    <input
            type="search"
            name="search"
            placeholder="Search Job Card or Customer..."
            value="<?= htmlspecialchars($search) ?>"
            aria-label="Search Job Card or Customer"
    />
    <button type="submit">Search</button>
</form>

<?php if ($message): ?>
    <div class="message" role="alert"><?= htmlspecialchars($message) ?></div>
<?php endif; ?>

<!-- Summary -->
<p><strong>
        Showing <?= $countRes['total'] ?> job cards,
        <?= $countRes['dispatched_count'] ?> dispatched,
        <?= $countRes['invoiced_count'] ?> invoiced
        <?php if ($selected_department !== 'All') echo " in " . htmlspecialchars($selected_department); ?>.
        Open: <?= $countRes['open_count'] ?>,
        Closed: <?= $countRes['closed_count'] ?>,
        Returned: <?= $countRes['returned_count'] ?>,
        Due: <?= $countRes['due_count'] ?>
    </strong></p>


<?php if (count($job_cards) === 0): ?>
    <p>No job cards found.</p>
<?php else: ?>
    <table role="grid" aria-label="Job Cards Table">
        <thead>
        <tr>
            <th scope="col">Job Card Number</th>
            <th scope="col">Customer</th>
            <th scope="col">Description</th>
            <th scope="col">Department</th>
            <th scope="col">Stage(s)</th>
            <th scope="col">Due Date</th>
            <th scope="col">Status</th>
            <th scope="col">Invoice Number</th>
        </tr>
        </thead>
        <tbody>
        <?php foreach ($job_cards as $job):
            $class = statusClass($job);
            $display_status = $job['status'] ?: (($job['dispatched']) ? 'dispatched' : 'open');
            ?>
            <tr class="<?= $class ?>">
                <td><?= htmlspecialchars($job['job_card_number']) ?></td>
                <td><?= htmlspecialchars($job['customer_name']) ?></td>
                <td><?= htmlspecialchars($job['order_description']) ?></td>
                <td><?= htmlspecialchars($job['department']) ?></td>
                <td><?= htmlspecialchars($job['stages'] ?? '-') ?></td>
                <td><?= htmlspecialchars($job['due_date'] ?? '-') ?></td>
                <td><?= htmlspecialchars(ucfirst($display_status)) ?></td>
                <td>
                    <?= ($job['invoiced'] && !empty($job['invoice_number'])) ? htmlspecialchars($job['invoice_number']) : '-' ?>
                </td>
            </tr>
        <?php endforeach; ?>
        </tbody>
    </table>
<?php endif; ?>

</body>
</html>
